/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.bpel.tests.testcases;

import java.util.Comparator;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;

import junit.framework.Test;

import org.jboss.soa.bpel.tests.RiftSawTest;
import org.jboss.soa.bpel.tests.RiftSawTestSetup;
import org.mortbay.util.ajax.JSON;

/**
 * Test case for running the RIFTSAW-155 JIRA validation.
 * See https://jira.jboss.org/browse/RIFTSAW-155
 */
public class RiftSaw155TestCase extends RiftSawTest {

	private static final String TEST_NAME = "RiftSaw_155";

	public RiftSaw155TestCase() {
		super(TEST_NAME);
	}
	
	public static Test suite() {
//		String test_dir = System.getProperty("test.dir");
//		System.setProperty("test.dir", "/home/bbrodt/workspace/test/target/tests");
//		System.setProperty("riftsaw.test.deployment.properties", "/home/bbrodt/jboss/riftsaw-2.0.1-SNAPSHOT/install/deployment.properties");
//		System.setProperty("java.naming.factory.initial", "org.jnp.interfaces.NamingContextFactory");
//		System.setProperty("java.naming.factory.url.pkgs", "org.jboss.naming:org.jnp.interfaces");
//		System.setProperty("java.naming.provider.url", "jnp://localhost:1099");
//		System.setProperty("java.rmi.server.ignoreStubClasses", "true");
		
		return(new RiftSawTestSetup(RiftSaw155TestCase.class,
						TEST_NAME, "RiftSaw_155-1.jar RiftSaw_155-2.jar RiftSaw_155-3.jar"));
	}

	private static class ProcInfo {
		public ProcInfo(String name,String id,boolean suspended)
		{
			this.name = name;
			this.id = id;
			this.suspended = suspended;
		}
		public String name;
		public String id;
		public boolean suspended;
	}
	
	private class ProcNameCompare implements Comparator
    {
		public int compare(Object arg1, Object arg2) {
			String name1 = arg1.toString();
			String name2 = arg2.toString();
			String namepart1 = name1.substring(0, name1.indexOf("-"));
			String namepart2 = name2.substring(0, name2.indexOf("-"));
			Integer versionpart1 = Integer.parseInt( name1.substring(name1.indexOf("-")+1) );
			Integer versionpart2 = Integer.parseInt( name2.substring(name2.indexOf("-")+1) );
			if ( namepart1.equals(namepart2) )
				return versionpart1.compareTo(versionpart2);
			return namepart1.compareTo(namepart2);
		}
    }
	
	public void testSendHello() throws Exception {
		try
		{
			String result = null;
			
			// we should have 3 versions each of "hello" and "goodbye" services
			// which were deployed in order. send an initial test message to check
			// that version 3 is the active one
			result = sendAndValidate("http://localhost:8080/hello", "hello_request1.xml", "hello_response3.xml");
			
			TreeMap<String,ProcInfo> helloProcs = getProcesses("hello");
			TreeMap<String,ProcInfo> goodbyeProcs = getProcesses("goodbye");
			
			// make sure there is only one active version of each process
			validateActiveProcesses(helloProcs,1);
			validateActiveProcesses(goodbyeProcs,1);

			// retire only version 3 of the "hello" process which should make version 2 active
			// NOTE: version 3 is the last entry in the sorted TreeMap
			retireProcess(helloProcs.get(helloProcs.lastKey()).id);
			
			// at this point there should be NO active versions of "hello"
			result = sendAndValidate("http://localhost:8080/hello", "hello_request1.xml", "hello_fail_response1.xml");
			
			// activate "hello" version 1
			// NOTE: this is the first entry in the TreeMap
			activateProcess(helloProcs.get(helloProcs.firstKey()).id);
			
			// check that version 1 of "hello" is active
			result = sendAndValidate("http://localhost:8080/hello", "hello_request1.xml", "hello_response1.xml");
			// and that there is only one version of each process
			validateActiveProcesses(helloProcs,1);
			validateActiveProcesses(goodbyeProcs,1);
			
			// make sure we still have version 3 of "goodbye"
			result = sendAndValidate("http://localhost:8080/goodbye", "goodbye_request1.xml", "goodbye_response3.xml");

			// undeploy version 1 of "hello" - there should be NO active versions of this process
			undeploy("RiftSaw_155-1.jar");
			validateActiveProcesses(helloProcs,0);
			validateActiveProcesses(goodbyeProcs,1);
			// also remove it from our TreeMap
			helloProcs.remove(helloProcs.firstKey());
			
			// but "goodbye" should still be active
			result = sendAndValidate("http://localhost:8080/goodbye", "goodbye_request1.xml", "goodbye_response3.xml");
			
			// activate version 2 of "hello"
			// NOTE: this is the first entry in the TreeMap
			activateProcess(helloProcs.get(helloProcs.firstKey()).id);

			// check that version 2 of "hello" is active
			result = sendAndValidate("http://localhost:8080/hello", "hello_request1.xml", "hello_response2.xml");
			// and that there is only one version of each process
			validateActiveProcesses(helloProcs,1);
			validateActiveProcesses(goodbyeProcs,1);
			
			// THE END!
		}
		catch(Exception ex)
		{
			fail(ex.getMessage());
		}
	}
	
	private TreeMap<String,ProcInfo> getProcesses(String name) throws Exception
	{
		TreeMap<String,ProcInfo> procs = new TreeMap<String,ProcInfo>();
		
		// request the processes currently deployed - should only be "hello" and "goodbye"
		String result = sendHttpGETMessage("http://localhost:8080/gwt-console-server/rs/engine/deployments");
		Map root = (Map)JSON.parse(result);
		Object[] deployments = (Object[])root.get("deployments");
		for ( Object item : deployments)
		{
			Map dep = (Map)item;
			ProcInfo proc = new ProcInfo(
					dep.get("name").toString(),
					dep.get("id").toString(),
					(Boolean)dep.get("suspended")
			);
			if (proc.name.contains(name))
				procs.put(proc.name,proc);
		}
		return procs;
	}
	
	private void validateActiveProcesses(TreeMap<String,ProcInfo> procs, int expected)
	{
		// make sure there is only one active version of each process
		String name = "unknown";
		int count = 0;
		for ( Entry<String, ProcInfo> p : procs.entrySet() )
		{
			name = p.getKey();
			if (p.getValue().suspended == false)
				++count;
		}
		if ( count!=expected )
			fail("The number of active '"+name+"' processes is "+count+" instead of "+expected);
	}
	
	private void retireProcess(String pid) throws Exception
	{
		sendHttpPOSTMessage("http://localhost:8080/gwt-console-server/rs/engine/deployment/" +
				pid +
				"/suspend");
	}
	
	private void activateProcess(String pid) throws Exception
	{
		sendHttpPOSTMessage("http://localhost:8080/gwt-console-server/rs/engine/deployment/" +
				pid +
				"/resume");
	}
}