/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.soa.bpel.clustering;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.ode.bpel.iapi.ClusterAware;
import org.apache.ode.bpel.iapi.Scheduler;
import org.apache.ode.dao.scheduler.DatabaseException;
import org.apache.ode.dao.scheduler.SchedulerDAOConnection;
import org.apache.ode.dao.scheduler.SchedulerDAOConnectionFactory;
import org.jboss.ha.framework.interfaces.HAPartition.HAMembershipListener;

/**
 * 
 * 
 * @author Jeff Yu
 *
 */
public class ODEJobClusterListener implements HAMembershipListener{
	
	private static Log logger = LogFactory.getLog(ODEJobClusterListener.class);
	
	private SchedulerDAOConnectionFactory schedulerCF;
	
	private Scheduler scheduler;
	
	private TransactionManager txm;
	
	public ODEJobClusterListener(Scheduler scheduler, SchedulerDAOConnectionFactory scf, TransactionManager tmgr) {
		this.scheduler = scheduler;
		this.schedulerCF = scf;
		this.txm = tmgr;
	}
	
	
	/**
	 * Move jobs that were associated with dead members to an active node.
	 * 
	 */
	public void membershipChanged(Vector deadMembers, Vector newMembers, Vector allMembers) {
		if (allMembers.isEmpty()) {
			throw new ClusteringException("There are no active nodes in this clustering environment");
		}
		
		if (scheduler instanceof ClusterAware) {
			setClusterNodeListsIntoScheduler(allMembers);
		}
		
		try {
			txm.begin();
			if (!deadMembers.isEmpty()) {
				SchedulerDAOConnection conn = schedulerCF.getConnection();
				String activeNodeId = allMembers.iterator().next().toString();
				Iterator<?> it = deadMembers.iterator();
				try {
					while (it.hasNext()) {
						String deadNodeId = it.next().toString();
						conn.updateReassign(deadNodeId, activeNodeId);
						logger.debug("Move all of jobs associated with [" + deadNodeId + "] to new node [" + activeNodeId + "]");
					}				
				} catch (DatabaseException e) {
					String errMsg = "Error in updating Job from dead node to active node : " + activeNodeId;
					logger.error(errMsg);
					throw new ClusteringException(errMsg, e);
				}

			}
			txm.commit();
		} catch (Exception e) {
			try {
				txm.rollback();
			} catch (Exception e1) {
				e1.printStackTrace();
				logger.error(e1);
			}
		}			
		
	}

	
	private void setClusterNodeListsIntoScheduler(Vector allMembers) {
		List<String> nodeLists = new ArrayList<String>();
		Iterator<?> nodes = allMembers.iterator();
		while (nodes.hasNext()) {
			String node = nodes.next().toString();
			nodeLists.add(node);
		}
		((ClusterAware)scheduler).setNodeList(nodeLists);
		logger.info("The available nodes now are " + nodeLists);
	}

}
