/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.services.soapui;

import java.io.IOException;
import java.util.Map;
import java.util.Properties;

import org.jboss.system.ServiceMBean;
import org.xml.sax.SAXException;

/**
 * Soap UI Soap Client Service.
 * <p/>
 * Embeds the <a href="http://www.soapui.org/">soapui</a> tool, using it to generate
 * a valid soap requests, as well as unmarshal response data from a valid soap response.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public interface SoapUIClientServiceMBean extends ServiceMBean {

    /**
     * Build a SOAP request for the specified operation on the specified WSDL.
     *
     * @param wsdl            WSDL URL.
     * @param operation       Operation name.
     * @param serviceName     Service Name.
     * @param params          Message parameter map.
     * @param httpClientProps {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @param smooksResource  {@link org.milyn.Smooks} transformation configuration resource file.
     *                        Null if no transformations are to be performed on the SOAP message before serializing it
     *                        for return.
     * @param soapNs 		  optional SOAP namespace
     * @return The SOAP Message.
     * @throws IOException                   Failed to load WSDL.
     * @throws UnsupportedOperationException Operation not supported on specified WSDL.
     * @throws SAXException                  Failed to parse the SOAP UI generated request message.
     */
    public abstract String buildRequest(String wsdl, String operation, String serviceName, Map params, Properties httpClientProps, String smooksResource, String soapNs) throws IOException, UnsupportedOperationException, SAXException;

    /**
     * Use soapUI mockService to Build a SOAP response for the specified operation on the specified WSDL.
     *
     * @param wsdl            WSDL URL.
     * @param operation       Operation name.
     * @param serviceName     Service Name.
     * @param params          Message parameter map..
     * @param smooksResource  {@link org.milyn.Smooks} transformation configuration resource file.
     *                        Null if no transformations are to be performed on the SOAP message before serializing it
     *                        for return.
     * @param soapNs 		  optional SOAP namespace
     * @return The SOAP Message.
     * @throws IOException                   Failed to load WSDL.
     * @throws UnsupportedOperationException Operation not supported on specified WSDL.
     * @throws SAXException                  Failed to parse the SOAP UI generated request message.
     */
    public abstract String buildResponse(String wsdl, String operation, String serviceName, Map params, Properties httpClientProps, String smooksResource, String soapNs) throws IOException, UnsupportedOperationException, SAXException;

    
    public abstract String buildFault(String wsdl, String operation, String serviceName, String faultName, Map params, Properties httpClientProps, String smooksResource, String soapNs) throws IOException, UnsupportedOperationException, SAXException;
    
    /**
     * Get the named endpoint from the specified WSDL.
     *
     * @param wsdl WSDL URL.
     * @param serviceName     Service Name.
     * @param httpClientProps {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @return The operation endpoint URL.
     * @throws IOException Failed to load WSDL.
     */
    public abstract String getEndpoint(String wsdl, String serviceName, Properties httpClientProps) throws IOException;
    
    /**
     * Get the Content Type for the appropriate SOAP version interface (for the named service) from the specified WSDL.
     *
     * @param wsdl WSDL URL.
     * @param serviceName     Service Name.
     * @param httpClientProps {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @return The operation endpoint URL.
     * @throws IOException Failed to load WSDL.
     */
    public abstract String getContentType(String wsdl, String serviceName, Properties httpClientProps) throws IOException;
    
    /**
     * Use soapUI mockService to Merge a SOAP response for the specified operation on the specified WSDL with its template.
     *
     * @param wsdl            WSDL URL.
     * @param operation       Operation name.
     * @param serviceName     Service Name.
     * @param response        The actual response.
     * @param smooksResource  {@link org.milyn.Smooks} transformation configuration resource file.
     *                        Null if no transformations are to be performed on the SOAP message before serializing it
     *                        for return.
     * @param soapNs 		  optional SOAP namespace
     * @return The SOAP Message.
     * @throws IOException                   Failed to load WSDL.
     * @throws UnsupportedOperationException Operation not supported on specified WSDL.
     * @throws SAXException                  Failed to parse the SOAP UI generated request message.
     */
    public abstract String mergeResponseTemplate(String wsdl, String operation, String serviceName, String response, Properties httpClientProps, String smooksResource, String soapNs) throws IOException, UnsupportedOperationException, SAXException;

    /**
     * Get the property file.
     * @return The name of the property file being used.
     */
    public String getPropertyFile() ;
    /**
     * Set the property file.
     * @param propertyFile The name of the property file being used.
     */
    public void setPropertyFile(final String propertyFile) ;
    
    /**
     * Get the additional schema resources.
     * @return The additional schema resources.
     */
    public String getSchemas() ;
    
    /**
     * Set the additional schemes.
     * @param schemas The additional schema resources to setup for SoapUI.
     * This needs support through the soapui-settings.xml file
     * @see setPropertyFile
     */
    public void setSchemas(final String schemas) ;
    
    public String getServerDataDir() ;
    public void setServerDataDir(String dataDir) ;
    
}
