package org.jboss.soa.esb.actions.converters;
/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */


import java.util.HashMap;
import java.util.List;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.DynamicMBean;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.apache.log4j.Logger;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.listeners.ListenerTagNames;

/**
 * Smooks message counter is a MBean that displays shows
 * how many failed and successful messages have been processed and shows the processing time
 * of each.
 * 
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 * @since Version 4.2
 */
public class SmooksMessageCounter implements DynamicMBean {
	private static final Logger logger = Logger.getLogger(SmooksMessageCounter.class);
	public static final String TRANSFORM_SUCCEED = "SUCCEED";
	public static final String TRANSFORM_FAILED = "FAILED";
	
	private HashMap<String, Integer> actionCounterHash;
	private HashMap<String, Integer> actionFailedCounterHash;
	private HashMap<String, Long> actionProcessTimeHash;
	private ConfigTree m_config;
	private Integer serviceCount;
	
	public static final String RESET_COUNTER = "resetCounter";
	private static final String MESSAGE_COUNTER = "messages successfully processed count";
	private static final String FAILED_MESSAGE_COUNTER = "messages failed count";
	private static final String PROCESSING_TIME = "processing time";
	private static final String OVERALL_SERVICE_COUNT = "overall message transform count";
	
	/**
	 * Constructor
	 * @param f_config config tree
	 */
	public SmooksMessageCounter(ConfigTree f_config) {
		
		actionCounterHash = new HashMap<String, Integer>();
		actionFailedCounterHash = new HashMap<String, Integer>();
		actionProcessTimeHash = new HashMap<String, Long>();
		serviceCount = Integer.valueOf(0);
		
		m_config = f_config;
	}
		
	/**
	 * Increment the total message count of this service.
	 */
	public void incrementTotalCount() {
		serviceCount = new Integer(serviceCount.intValue() + 1);
	}
	
	/**
	 * Initialize the hashes by setting the counts and the processing time to an initial value of 0. 
	 */
	public void initHashes() {
		serviceCount = Integer.valueOf(0);
	}
	
	/**
	 * Reset the counters - set all the entries in the action counter hash 
	 * and in the action process time hash to zero.
	 */
	public void resetCounter() {
		serviceCount = Integer.valueOf(0);
		
		for (String key : actionCounterHash.keySet()) {
			actionCounterHash.put(key, Integer.valueOf(0));
		}
		
		for (String key : actionFailedCounterHash.keySet()) {
			actionFailedCounterHash.put(key, Integer.valueOf(0));
		}
		
		for (String key : actionProcessTimeHash.keySet()) {
			actionProcessTimeHash.put(key, Long.valueOf(0));
		}
	}
	
	/**
	 *  This creates the MBeanInfo object provided.     We are returning generic 
	 *  text for the attribute descriptions (the word Property and the name of the 
	 *  attribute), all of the attributes are read-only, and we provide four 
	 *  invocation methods - start/stop/initialise/destroy on the Lifecycle. 
	 */
    public MBeanInfo getMBeanInfo() {
		
		int count = actionCounterHash.size() + actionProcessTimeHash.size()
			+ actionFailedCounterHash.size() + 1;
        MBeanAttributeInfo[] attrs = new MBeanAttributeInfo[count];
        int counter = 0;
              
		for (String key : actionCounterHash.keySet()) {
            attrs[counter] = new MBeanAttributeInfo(
                    key, "java.lang.Integer", "Property " + key, true, false, false);
            counter++;
		}
		
		for (String key : actionProcessTimeHash.keySet()) {
            attrs[counter] = new MBeanAttributeInfo(
                    key, "java.lang.Double", "Property " + key, true, false, false);
            counter++;
		}
		
		for (String key : actionFailedCounterHash.keySet()) {
            attrs[counter] = new MBeanAttributeInfo(
                    key, "java.lang.Integer", "Property " + key, true, false, false);
            counter++;
		}
		
		MBeanAttributeInfo overallCount = new MBeanAttributeInfo(OVERALL_SERVICE_COUNT, "java.lang.Integer",
				"Property " + OVERALL_SERVICE_COUNT, true, false, false);
		attrs[counter] = overallCount;
		counter++;
			
        MBeanOperationInfo[] opers = {
        	new MBeanOperationInfo(
        			RESET_COUNTER, "Reset the counter",
                	null, "void", MBeanOperationInfo.ACTION)
        };
        return new MBeanInfo(
                this.getClass().getName(), "Service Message Counter MBean",
                attrs, null, opers, null); // notifications
	}

	/* (non-Javadoc)
	 * @see javax.management.DynamicMBean#getAttribute(java.lang.String)
	 */
	public Object getAttribute(String key) throws AttributeNotFoundException, MBeanException, ReflectionException {
    	if (actionCounterHash.containsKey(key)) {
    		Integer value = null;
    		value = actionCounterHash.get(key);
    		return value;
    	} else if (actionProcessTimeHash.containsKey(key)) {
    		Long processTotal = actionProcessTimeHash.get(key);
    		String actionId = key.substring(0, key.indexOf(PROCESSING_TIME)-1);
    		Integer successCount = actionCounterHash.get(actionId + " " + MESSAGE_COUNTER);
    		Double value = null;
    		
    		if (successCount.intValue() > 0) {
    			value = ((double) processTotal / successCount.intValue());
    		} else {
    			value = null;
    		}
    		return value;
    	} else if (actionFailedCounterHash.containsKey(key)) {
    		Integer value = actionFailedCounterHash.get(key);
    		return value;
    	} else if (OVERALL_SERVICE_COUNT.equals(key)) {
    		return serviceCount;
    	}
    	return null;
    }

	/* (non-Javadoc)
	 * @see javax.management.DynamicMBean#getAttributes(java.lang.String[])
	 */
	public AttributeList getAttributes(String[] arg0) {
		AttributeList attributeList = new AttributeList();
		for (String key : actionCounterHash.keySet()) {
			Attribute at = new Attribute(key, actionCounterHash.get(key).toString());
			attributeList.add(at);
		}
		
		Attribute overallCount = new Attribute(OVERALL_SERVICE_COUNT, serviceCount);
		attributeList.add(overallCount);
		
		for (String key : actionProcessTimeHash.keySet()) {
			Long processTotal = actionProcessTimeHash.get(key);
			String actionId = key.substring(0, key.indexOf(PROCESSING_TIME));
			Integer successCount = actionCounterHash.get(actionId + " " + MESSAGE_COUNTER);
			String avgTime = null;
			if (successCount.intValue() > 0) {
				avgTime = ((double) processTotal / successCount.intValue()) + " ns";
			}
			Attribute at = new Attribute(key, avgTime);
			attributeList.add(at);
		}
		
		for (String key : actionFailedCounterHash.keySet()) {
			Attribute at = new Attribute(key, actionFailedCounterHash.get(key).toString());
			attributeList.add(at);
		}

		return attributeList;
	}

	/* (non-Javadoc)
	 * @see javax.management.DynamicMBean#invoke(java.lang.String, java.lang.Object[], java.lang.String[])
	 */
	public Object invoke(String method, Object[] arg1, String[] arg2) throws MBeanException, ReflectionException {
		if (method.equalsIgnoreCase(RESET_COUNTER)) {
				resetCounter();
			return "Invoking the " + method + " on the lifecycle.";
		} else {
			throw new ReflectionException(new NoSuchMethodException(method));
		}
	}

	/* (non-Javadoc)
	 * @see javax.management.DynamicMBean#setAttribute(javax.management.Attribute)
	 */
	public void setAttribute(Attribute arg0) throws AttributeNotFoundException, InvalidAttributeValueException, MBeanException, ReflectionException {
	}

	/* (non-Javadoc)
	 * @see javax.management.DynamicMBean#setAttributes(javax.management.AttributeList)
	 */
	public AttributeList setAttributes(AttributeList arg0) {
		return null;
	}

	/**
	 * Register this MBean with JBoss.
	 */
	protected void registerMBean() {
		MBeanServer mbeanServer = null;
		try {
			mbeanServer = MBeanServerLocator.locateJBoss();
		} catch (IllegalStateException ise) {
			// If we can't find a JBoss MBeanServer, just return
			// Needed for unit tests
			return;
		}
		
		ObjectName listObjectName = getObjectName();
		
		if (mbeanServer.isRegistered(listObjectName)) {
	    	try {
	    		mbeanServer.unregisterMBean(listObjectName);
			} catch (InstanceNotFoundException e) {
				e.printStackTrace();
			} catch (MBeanRegistrationException e) {
				e.printStackTrace();
			}
        }
		
	    try {
	    	mbeanServer.registerMBean(this, listObjectName);
		} catch (InstanceAlreadyExistsException e) {
			e.printStackTrace();
		} catch (MBeanRegistrationException e) {
			e.printStackTrace();
		} catch (NotCompliantMBeanException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Initialize the message profile with a count of 0 for the time hash, the processed hash, and the failed hash.
	 * @param messageProfile message profile
	 */
	public void initMessageProfile(String messageProfile) {
		String processTimeKey = messageProfile + " " + PROCESSING_TIME;
		if (!actionProcessTimeHash.containsKey(processTimeKey)) {
			actionProcessTimeHash.put(processTimeKey, Long.valueOf(0));
		}
		String messageCounterKey = messageProfile + " " + MESSAGE_COUNTER;
		if (!actionCounterHash.containsKey(messageCounterKey)) {
			actionCounterHash.put(messageCounterKey, Integer.valueOf(0));
		}
		String failedCounterKey = messageProfile + " " + FAILED_MESSAGE_COUNTER;
		if (!actionFailedCounterHash.containsKey(failedCounterKey)) {
			actionCounterHash.put(failedCounterKey, Integer.valueOf(0));
		}
	}
	

	/**
	 * Update the counts of the SmooksMessageCounter for a particular message profile.
	 * @param procTime time
	 * @param messageProfile profile
	 * @param status status
	 */
	public void update(long procTime, String messageProfile, String status) {		
		if ((messageProfile != null) && (!messageProfile.equals(""))) {
			initMessageProfile(messageProfile);
		}
		
		if (TRANSFORM_SUCCEED.equals(status)) {
			Integer count = null;
			Long time = null;
			if (actionCounterHash.containsKey(messageProfile + " " + MESSAGE_COUNTER)) {
				count = actionCounterHash.get(messageProfile + " " + MESSAGE_COUNTER);
				count = count.intValue() + 1;
			} else {
				count = new Integer(1);
			}
			actionCounterHash.put(messageProfile + " " + MESSAGE_COUNTER, count);
			
			if (actionProcessTimeHash.containsKey(messageProfile + " " + PROCESSING_TIME)) {
				time = actionProcessTimeHash.get(messageProfile + " " + PROCESSING_TIME);
				time = time.longValue() + procTime;				
			} else {
				time = procTime;
			}
			actionProcessTimeHash.put(messageProfile + " " + PROCESSING_TIME, time);
		} else if (TRANSFORM_FAILED.equals(status)) {
			Integer count = null;
			if (actionFailedCounterHash.containsKey(messageProfile + " " + FAILED_MESSAGE_COUNTER)) {
				count = actionFailedCounterHash.get(messageProfile + " " + FAILED_MESSAGE_COUNTER);
				count = count.intValue() + 1; 
			} else {
				count = new Integer(1);
			}
			actionFailedCounterHash.put(messageProfile + " " + FAILED_MESSAGE_COUNTER, count);
		}		
	}
	
	/**
	 * Return the object name, which for a SmooksMessageCounter is the deployment name (x.esb) and the category
	 * (SmooksMessageCounter).
	 * @return object name
	 */
	protected ObjectName getObjectName()
	{
		String deploymentName = "";
		List<KeyValuePair> properties = m_config.attributesAsList();
		
		// Get Deployment Name
		try {
			deploymentName = m_config.getParent().getParent().getAttribute("deployment");
		} catch (Exception e) {
			logger.error("", e);
		}
		
		ObjectName listObjectName = null;
		try {

			StringBuffer objectName = new StringBuffer("category=SmooksMessageCounter");
			if (deploymentName != null) {
				objectName.append(",").append(ListenerTagNames.DEPLOYMENT_NAME_TAG).append("=").append(deploymentName);
			}
			
			listObjectName = new ObjectName("jboss.esb:" + objectName.toString());
		} catch (MalformedObjectNameException e1) {
			e1.printStackTrace();
		} catch (NullPointerException e1) {
			e1.printStackTrace();
		}
		return listObjectName;
	}
}
