/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions;

import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.InputStream;
import java.net.URI;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.testutils.FileUtil;
import org.jboss.soa.esb.util.ClassUtil;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Test the jBPM command interpreter class
 *
 * @author <a href="mailto:kurt.stam@jboss.com">Kurt Stam</a>
 */


public class MessageFilterUnitTest
{
	static Logger 				_logger = Logger.getLogger(MessageFilterUnitTest.class);
    static Message message = null;
    
    private static EPR epr1;
    private static EPR epr2;
    private static EPR epr3;
    private static MockCourier courier1;
    private static MockCourier courier2;
    private static MockCourier courier3;
    
    private static ConfigTree[] actions;
	
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(MessageFilterUnitTest.class);
    }
    
    
    @BeforeClass
    public static void before() throws Exception {
        MockCourierFactory.install();
        MockRegistry.install();

        epr1 = new EPR(new URI("test1"));
        epr2 = new EPR(new URI("test2"));
        epr3 = new EPR(new URI("DLS"));
        courier1 = new MockCourier(true);
        courier2 = new MockCourier(true);
        courier3 = new MockCourier(true);

        MockRegistry.register("test", "java", epr1, courier1);
        MockRegistry.register("test", "xml", epr2, courier2);
        MockRegistry.register(ServiceInvoker.INTERNAL_SERVICE_CATEGORY, ServiceInvoker.DEAD_LETTER_SERVICE_NAME, epr3, courier3);
 
        message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
        message.getBody().add(("Hello CBR"));
        
        InputStream in = ClassUtil.getResourceAsStream("ContentBasedRouterUnitTest.xml", MessageFilterUnitTest.class);
        String xml = FileUtil.readStream(in);
        actions = ConfigTree.fromXml(xml).getChildren("action");
    }
    
    @Test
	public void wrongDestinationName()
    {
		try {
            MessageFilter cbr = new MessageFilter(actions[0]);
            cbr.process(message);
            //the config is wrong so we should error here, and try to send the message to the DLS
            assertTrue(false);
		} catch (Exception e) {
            /* As it can't send the message to the DLS the log should say:
               ERROR [main][ContentBasedWiretap] Destination serialized-destination does not exist your configuration
               ERROR [main][ContentBasedWiretap] No rule destination(s) were matched, [serialized, xml]. Please fix your configuration and/or routing rules.
             */
			assertTrue(true);
		}
    }
    
    @Test
    public void correctDestinationName()
    {
        try {
            MessageFilter cbr = new MessageFilter(actions[1]);
            Message returnMessage = cbr.process(message);
            //the cbr should return a null message
            assertNull(returnMessage);
        } catch (Exception e) {
            assertTrue(false);
        }
    }
    
    @Test
    public void noDestinationMatches()
    {
        try {
            MessageFilter cbr = new MessageFilter(actions[2]);
            Message returnMessage = cbr.process(message);
            //the cbr should return a null message, message should be filtered with no errors.
            assertNull(returnMessage);
        } catch (Exception e) {
            assertTrue(false);
        }
    }
}