/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.rules;

/**
 * This enum contains properties specific for a RuleService.
 * <p/>
 * 
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 * @author dward at jboss.org
 */
public enum RuleServicePropertiesNames {
	
	IMPL_CLASS ( "ruleServiceImplClass", null ),
	DECISION_TABLE ( "decisionTable", null ),
	RULE_AGENT_PROPERTIES ( "ruleAgentProperties", null ),
	RULE_AUDIT_TYPE ( "ruleAuditType", new StringValue[]{StringValue.CONSOLE, StringValue.FILE, StringValue.THREADED_FILE} ),
	RULE_AUDIT_FILE ( "ruleAuditFile", null ),
	RULE_AUDIT_INTERVAL ( "ruleAuditInterval", null ),
	RULE_CLOCK_TYPE ( "ruleClockType", new StringValue[]{StringValue.REALTIME, StringValue.PSEUDO}),
	RULE_EVENT_PROCESSING_TYPE ( "ruleEventProcessingType", new StringValue[]{StringValue.STREAM, StringValue.CLOUD}),
	RULE_FIRE_METHOD ( "ruleFireMethod", new StringValue[]{StringValue.FIRE_ALL_RULES, StringValue.FIRE_UNTIL_HALT} ),
	RULE_MAX_THREADS ( "ruleMaxThreads", null ),
	RULE_MULTITHREAD_EVALUATION ( "ruleMultithreadEvaluation", new BooleanValue[]{BooleanValue.TRUE, BooleanValue.FALSE} ),
	DISPOSE ( "dispose", new BooleanValue[]{BooleanValue.TRUE, BooleanValue.FALSE} ),
	CONTINUE ( "continue", new BooleanValue[]{BooleanValue.TRUE, BooleanValue.FALSE} ),
	STATEFUL ( "stateful", new BooleanValue[]{BooleanValue.TRUE, BooleanValue.FALSE} ),
    DEFAULT_CONTINUE ( "defaultContinue", new BooleanValue[]{BooleanValue.TRUE, BooleanValue.FALSE}) ;
	
	private final String name;
	private final Value<?>[] values;

	private <T> RuleServicePropertiesNames( final String name, final Value<T>[] values )
	{
		this.name = name;
		this.values = (values != null ? values : new Value<?>[]{});
	}
	
	public String getName()
	{
		return name;
	}
	
	public Value<?> getValue(String trial)
	{
		for (Value<?> value : values)
		{
			if (value.equivalent(trial))
			{
				return value;
			}
		}
		return null;
	}
	
	public BooleanValue getBooleanValue(String trial)
	{
		Value<?> value = getValue(trial);
		if (value instanceof BooleanValue) {
			return (BooleanValue)value;
		}
		return null;
	}
	
	public StringValue getStringValue(String trial)
	{
		Value<?> value = getValue(trial);
		if (value instanceof StringValue) {
			return (StringValue)value;
		}
		return null;
	}
	
	public interface Value<T> {
		T get();
		boolean equivalent(String s);
	}
	
	public static enum BooleanValue implements Value<Boolean> {
		TRUE(Boolean.TRUE),
		FALSE(Boolean.FALSE);
		private Boolean get;
		private BooleanValue(Boolean get) {
			this.get = get;
		}
		public Boolean get() {
			return this.get;
		}
		public boolean equivalent(String s) {
			if (s != null) {
				s = s.trim();
			}
			return get().toString().equalsIgnoreCase(s);
		}
	}
	
	public static enum StringValue implements Value<String> {
		NULL(null),
		EMPTY(""),
		CONSOLE("CONSOLE"),
		FILE("FILE"),
		THREADED_FILE("THREADED_FILE"),
		REALTIME("REALTIME"),
		PSEUDO("PSEUDO"),
		STREAM("STREAM"),
		CLOUD("CLOUD"),
		FIRE_ALL_RULES("FIRE_ALL_RULES"),
		FIRE_UNTIL_HALT("FIRE_UNTIL_HALT");
		private String get;
		private StringValue(String get) {
			this.get = get;
		}
		public String get() {
			return this.get;
		}
		public boolean equivalent(String s) {
			String got = get();
			if (got == null) {
				return (s == null);
			}
			if (s != null) {
				s = s.trim();
			}
			return got.equalsIgnoreCase(s);
		}
	}
	
}
