package org.jboss.soa.esb.services.jbpm5.actions;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import java.io.IOException;
import java.io.Serializable;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SealedObject;
import javax.crypto.SecretKey;
import javax.naming.Context;
import javax.naming.InitialContext;

import org.jboss.soa.esb.services.jbpm5.MockDataSource;
import org.jboss.soa.esb.services.jbpm5.MockTransactionManager;
import org.jboss.soa.esb.services.jbpm5.MockUserTransaction;
import org.jboss.soa.esb.services.jbpm5.actions.Bpm5Processor;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.security.SecurityContext;
import org.jboss.soa.esb.services.security.SecurityService;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.junit.Before;
import org.junit.Test;
import org.xml.sax.SAXException;

/**
 * Test the BPM5Process action's security context / auth request.
 *  
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 */
public class Bpm5ProcessorUnitTest {
    private Bpm5Processor bpmProcessor;
    private Message message;
    private ConfigTree configTree;

    @Test
    public void verifyThatSecurityContextWasAddedToMesssageContext() throws ConfigurationException, InvalidKeyException, NoSuchAlgorithmException, NoSuchPaddingException, IllegalBlockSizeException, IOException
    {
        final SealedObject securityContext = createSealedObject("dummy string");
        SecurityContext.setSecurityContext(securityContext);
        bpmProcessor.addSecurityContext(message);

        assertNotNull(message.getContext().getContext(SecurityService.CONTEXT));
        assertEquals(securityContext, message.getContext().getContext(SecurityService.CONTEXT));
    }

    @Test
    public void verifyThatNullSecurityContextCanBeAddedToMesssageContext() throws ConfigurationException, InvalidKeyException, NoSuchAlgorithmException, NoSuchPaddingException, IllegalBlockSizeException, IOException
    {
        SecurityContext.setSecurityContext(null);
        bpmProcessor.addSecurityContext(message);

        assertNull(message.getContext().getContext(SecurityService.CONTEXT));
    }

    @Test
    public void verifyThatAuthenticationRequestWasAddedToMessageContext() throws ConfigurationException
    {
        //  Doesn't need to be encrypted as this is only passed along and never descrypted.
        byte[] encryptedAuthRequest = new byte[100];
        AuthenticationRequestImpl.setEncryptedAuthRequest(encryptedAuthRequest);
        bpmProcessor.addAuthenticationRequest(message);

        assertNotNull(message.getContext().getContext(SecurityService.AUTH_REQUEST));
        assertEquals(encryptedAuthRequest, message.getContext().getContext(SecurityService.AUTH_REQUEST));
    }

    @Test
    public void verifyThatNullAuthenticationRequestCanBeAddedToMessageContext() throws ConfigurationException
    {
        AuthenticationRequestImpl.setEncryptedAuthRequest(null);
        bpmProcessor.addAuthenticationRequest(message);
        assertNull(message.getContext().getContext(SecurityService.AUTH_REQUEST));
    }

    @Before
    public void setup() throws ConfigurationException
    {
    	Context ctx = null;
        try {
            ctx = new InitialContext();
            MockDataSource mds = new MockDataSource();
            MockUserTransaction mut = new MockUserTransaction();
            MockTransactionManager mtm = new MockTransactionManager();
            ctx.rebind("UserTransaction", mut);
            ctx.rebind("java:jboss/datasources/jbpm5DS", mds);
            ctx.rebind("java:/TransactionManager", mtm);
        } catch (Exception e) {
            //fail(e.getMessage());
        }
    	
    	try {
    		configTree = ConfigTree.fromXml(
        	 "<action action=\"jbpm5\" class=\"org.jboss.soa.esb.services.jbpm5.actions.Bpm5Processor\" process-action=\"startProcess\" "
        	  + "process-definition-name=\"Evaluation.bpmn\" process-id=\"com.sample.evaluation\">"
              + "<mapping bpm=\"employee\" esb=\"employee\" value=\"krisv\"/>"
              + "<mapping bpm=\"reason\" esb=\"reason\" value=\"Yearly performance review\"/>"
              + "</action>");
    	} catch (SAXException se){
    		throw new ConfigurationException(se);
    	}
        bpmProcessor = new Bpm5Processor(configTree);
        message = MessageFactory.getInstance().getMessage();
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(Bpm5ProcessorUnitTest.class);
    }

    private SealedObject createSealedObject(final Serializable ser) throws NoSuchAlgorithmException, NoSuchPaddingException, IllegalBlockSizeException, IOException, InvalidKeyException
    {
        final KeyGenerator kpg = KeyGenerator.getInstance("TripleDES");
        kpg.init(112);
        final SecretKey secretKey = kpg.generateKey();
        final Cipher cipher = Cipher.getInstance("TripleDES");
        cipher.init(Cipher.ENCRYPT_MODE, secretKey);

        return new SealedObject(ser, cipher);
    }

}
