/*
 * JBoss, Home of Professinal Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.jbpm5.actions;

import java.util.Map;
import javax.crypto.SealedObject;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.addressing.helpers.EPRHelper;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.MarshalException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.jbpm5.model.ProcessConstants;
import org.jboss.soa.esb.services.jbpm5.model.ProcessType;
import org.jboss.soa.esb.services.security.SecurityContext;
import org.jboss.soa.esb.services.security.SecurityService;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;


import org.drools.runtime.process.ProcessInstance;

import org.drools.runtime.StatefulKnowledgeSession;

/**
 * The Bpm5Processor processes an action on a single JBPM process.
 *  
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 * @since Version 4.11
 */
public class Bpm5Processor extends AbstractBpm5Action {
    protected static Logger logger = Logger.getLogger(Bpm5Processor.class);

    protected String processId;    
    protected String processAction;
    
	public Bpm5Processor(ConfigTree configTree) throws ConfigurationException
	{
	    super(configTree, logger) ;

        this.processId = configTree.getRequiredAttribute(ProcessConstants.PROCESS_ID);

		// verify the process action
		this.processAction = configTree.getRequiredAttribute(ProcessConstants.PROCESS_ACTION);
		verifyProcessAction(processAction);
	}
	
    /**
     * Verify that the process action matches one of the valid process action values.
     * @param processAction process action
     * @throws ConfigurationException 
     */
    public void verifyProcessAction(String processAction) throws ConfigurationException 
    {
    	if ((ProcessConstants.ABORT_PROCESS_INSTANCE_TYPE.equals(processAction)) ||
    		(ProcessConstants.SIGNAL_EVENT_TYPE.equals(processAction)) ||
    		(ProcessConstants.START_PROCESS_TYPE.equals(processAction))) {
    		return;
    	} else {
    		throw new ConfigurationException("Process action " + processAction +
    			"does not match valid process actions (" + 
    			ProcessConstants.START_PROCESS_TYPE + ", " +
    			ProcessConstants.ABORT_PROCESS_INSTANCE_TYPE + ")");
    	}
    			
    }
	
	public Message process(Message message) throws ActionProcessingException {
		// Add security context to message context.
		addSecurityContext(message);
		// Add authentication request context to message context.
		addAuthenticationRequest(message);
		
        Long processInstanceId = null;
        ProcessInstance processInstance = null;
				
		ProcessType processType = ProcessType.fromAction((String)processAction);
		
	    final StatefulKnowledgeSession ksession = getSession(message);
	    int sessionId = ksession.getId();
		switch (processType) {
			case START_PROCESS:
				logger.debug(ProcessConstants.START_PROCESS_TYPE);				
				Map<String, Object> params = BpmParameterMapper.mapPropertiesToParams(message, configTree);
				try {
					params.remove(ProcessConstants.SESSION_ID);
					params.put(ProcessConstants.SESSION_ID, Integer.toString(sessionId));
				} catch (Exception e) {
					logger.error("", e);
				}
				processInstance = ksession.startProcess(processId, params);
	            processInstanceId = Long.valueOf(processInstance.getId());				

	            final Call call = message.getHeader().getCall() ;
	            setEPR(message.getBody(), ProcessConstants.REPLY_TO, getEPR(call.getReplyTo())) ;
	            setEPR(message.getBody(), ProcessConstants.FAULT_TO, getEPR(call.getFaultTo())) ;

                break;
			case SIGNAL_EVENT:
				logger.debug(ProcessConstants.SIGNAL_EVENT_TYPE);
                Map<String, Object> eventParams = BpmParameterMapper.mapPropertiesToParams(message, configTree);
                try {
       				eventParams.remove(ProcessConstants.SESSION_ID);
                	eventParams.remove(ProcessConstants.PROCESS_EVENT);
                	eventParams.put(ProcessConstants.PROCESS_EVENT, message.getProperties().getProperty(ProcessConstants.PROCESS_EVENT));
                	eventParams.put(ProcessConstants.SESSION_ID, Integer.toString(sessionId));
                } catch (Exception e) {                                 
                	logger.error("", e);
                }
                
                String processEventType = (String) message.getProperties().getProperty(ProcessConstants.PROCESS_EVENT_TYPE);
                processInstanceId = getProcessInstanceId(message);
                if (processInstanceId != null) {
                    ksession.signalEvent(processEventType, eventParams, processInstanceId.longValue());
                } else {
                    throw new ActionProcessingException(ProcessConstants.SIGNAL_EVENT_TYPE
                        + ": tried to signal event and process id was null");
                }
                break;
	        case ABORT_PROCESS_INSTANCE:
				logger.debug(ProcessConstants.ABORT_PROCESS_INSTANCE_TYPE);
                processInstanceId = getProcessInstanceId(message);

                if (processInstanceId != null) {
	                ksession.abortProcessInstance(processInstanceId.longValue());
                } else {
	                throw new ActionProcessingException(ProcessConstants.ABORT_PROCESS_INSTANCE_TYPE 
	                    + ": tried to abort and process id was null");
	            }

                final Call abortCall = message.getHeader().getCall() ;
                setEPR(message.getBody(), ProcessConstants.REPLY_TO, getEPR(abortCall.getReplyTo())) ;
                setEPR(message.getBody(), ProcessConstants.FAULT_TO, getEPR(abortCall.getFaultTo())) ;

                break;
		}
		
		// Set Session and ProcessInstance information
		message.getContext().setContext(ProcessConstants.SESSION_ID, sessionId);
		message.getContext().setContext(ProcessConstants.PROCESS_INSTANCE_ID, 
				processInstanceId);
		
		return message;
	}

	/**
	 * Return the session id from the context id.
	 * @param msg
	 * @return
	 */
	public int getSessionId(Message msg) {
		int id = -1;
		try {
			id = ((Integer)msg.getContext().getContext(ProcessConstants.SESSION_ID)).intValue();
		} catch (Exception e) {
		}
		return id;		
	}
	
	/**
	 * Return the process instance id from the message context.
	 * @param msg
	 * @return
	 */
	public Long getProcessInstanceId(Message msg) {
		Long id = null;
		try {
			id = (Long)msg.getContext().getContext(ProcessConstants.PROCESS_INSTANCE_ID);
		} catch (Exception e) {
		}
		return id;
	}
	
	/**
	 * Adds the current security context to the passed in message.
	 * The context is added, if set, to the message context keyed by {@link SecurityService#CONTEXT}.
	 *
	 * @param message The ESB message object.
	 */
    final void addSecurityContext(Message message)
    {
        AssertArgument.isNotNull(message, "message");

        SealedObject securityContext = SecurityContext.getSecurityContext();
        if (securityContext != null)
        {
            message.getContext().setContext(SecurityService.CONTEXT, securityContext);
        }
    }

	/**
	 * Adds the authentication request to the passed in message.
	 * The context is added, if set, to the message context keyed by {@link SecurityService#AUTH_REQUEST}.
	 *
	 * @param message The ESB message object.
	 */
    final void addAuthenticationRequest(Message message)
    {
        AssertArgument.isNotNull(message, "message");

        byte[] encryptedAuthRequest = AuthenticationRequestImpl.getEncryptedAuthRequest();
        if (encryptedAuthRequest != null)
        {
            message.getContext().setContext(SecurityService.AUTH_REQUEST, encryptedAuthRequest);
        }
    }    
    
    private void setEPR(final Body body, final String name, final String value)
    {
        if (value == null)
        {
            body.remove(name) ;
        }
        else
        {
            body.add(name, value) ;
        }
    }
    
	private String getEPR(final EPR epr)
	{
		if (epr == null) {
			return null ;
		}
		
		try {
			return EPRHelper.toXMLString(epr) ;
		} catch (final MarshalException me) {
			throw new IllegalArgumentException("Failed to marshall epr", me) ;
		}
	}	
}
