/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.services.jbpm.actions;

import javax.crypto.SealedObject;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycle;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.jbpm.Constants;
import org.jboss.soa.esb.services.jbpm.cmd.Command;
import org.jboss.soa.esb.services.jbpm.cmd.CommandExecutor;
import org.jboss.soa.esb.services.jbpm.cmd.MessageFacade;
import org.jboss.soa.esb.services.security.SecurityContext;
import org.jboss.soa.esb.services.security.SecurityService;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;

/**
*
* Processes a single jBPM command.
* Part of the message content will be taken from constructor time configuration
* (from the action configuration XML)
*
* @author <a href="mailto:schifest@heuristica.com.ar">schifest@heuristica.com.ar</a>
* @author <a href="mailto:kurt.stam@jboss.com">Kurt Stam</a>
* @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
*/
public class BpmProcessor implements ActionLifecycle
{
    protected MessageFacade messageFacade;
    protected ConfigTree configTree;
    private static Logger _logger = Logger.getLogger(BpmProcessor.class);

    public void destroy() throws ActionLifecycleException
    {
        _logger.info("destroy() invoked");
    }
    public void initialise() throws ActionLifecycleException
    {
        _logger.info("initialise() invoked - config="+configTree);
        try {
            messageFacade   = MessageFacade.getFacade(configTree);
        } catch (ConfigurationException ce) {
            throw new ActionLifecycleException(ce.getMessage(), ce);
        }
    }

	public BpmProcessor(ConfigTree configTree) throws ConfigurationException
	{
        this.configTree = configTree;
	}

	public Message process(Message message) throws ConfigurationException
	{
		// Add security context to message context.
		addSecurityContext(message);
		// Add authentication request context to message context.
		addAuthenticationRequest(message);

		message.getBody().add(Constants.COMMAND_CODE, messageFacade.getOpCode().toString());
		messageFacade.setJBPMContextParameters(message);

        CommandExecutor commandExecutor = CommandExecutor.getInstance();
        String commandString  = (String)message.getBody().get(Constants.COMMAND_CODE);
        Command command = commandExecutor.getCommand(commandString);
        command.execute(message);
        return message;
	}

	/**
	 * Adds the current security context to the passed in message.
	 * The context is added, if set, to the message context keyed by {@link SecurityService#CONTEXT}.
	 *
	 * @param message The ESB message object.
	 */
    final void addSecurityContext(Message message)
    {
        AssertArgument.isNotNull(message, "message");

        SealedObject securityContext = SecurityContext.getSecurityContext();
        if (securityContext != null)
        {
            message.getContext().setContext(SecurityService.CONTEXT, securityContext);
        }
    }

	/**
	 * Adds the authentication request to the passed in message.
	 * The context is added, if set, to the message context keyed by {@link SecurityService#AUTH_REQUEST}.
	 *
	 * @param message The ESB message object.
	 */
    final void addAuthenticationRequest(Message message)
    {
        AssertArgument.isNotNull(message, "message");

        byte[] encryptedAuthRequest = AuthenticationRequestImpl.getEncryptedAuthRequest();
        if (encryptedAuthRequest != null)
        {
            message.getContext().setContext(SecurityService.AUTH_REQUEST, encryptedAuthRequest);
        }
    }
}
