/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author daniel.brum@jboss.com
 */

package org.jboss.soa.esb.messagestore;

import static org.junit.Assert.assertTrue;

import java.io.File;
import java.net.URI;
import java.sql.Connection;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.apache.log4j.xml.DOMConfigurator;
import org.jboss.internal.soa.esb.persistence.format.MessageStoreFactory;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.persistence.manager.ConnectionManager;
import org.jboss.soa.esb.persistence.manager.ConnectionManagerFactory;
import org.jboss.soa.esb.services.persistence.MessageStore;
import org.jboss.soa.esb.testutils.HsqldbUtil;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * @author dbrum
 *
 */
public class MessageStoreUnitTest  {
	private static Logger logger = Logger.getLogger(MessageStoreUnitTest.class);

	private static Object server ;
    
	@Test
	public void testMultiClientLoad() throws Exception {
		int clients = 5;
		Thread[] xx = new Thread[clients];
		for (int i=0; i<clients; i++) {
			xx[i] = new Thread(new MessageStoreClient());
		}
		for (int i=0; i<clients; i++) {
			xx[i].start();
		}
		//clean up the threads
		for (int i=0; i<clients; i++) {
			xx[i] = null;
		}
	}
	
	@Test
	public void testSingleLoad() throws Exception {
		long startTime=System.currentTimeMillis();
		int requestors = 1000;	//number of persist calls to make
		List<URI> uriList = new ArrayList<URI>();
		//get the database store
		MessageStore store = MessageStoreFactory.getInstance().getMessageStore();
		//messages to persist
		Message msg[] = { MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML), 
						 MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED)};		
		//set some content inside the Messages
		for (int x=0; x<msg.length; x++) {
			msg[x].getBody().add("TEST BODY".getBytes());
			msg[x].getProperties().setProperty("prop"+x, "val"+x);
			msg[x].getAttachment().addItem(new String("TEST ATTACHMENT"));						
		}
		//loop through adding the messages
		Random rndMsg = new Random();
		for (int x=0; x<=requestors; x++) {
			URI uid = store.addMessage(msg[rndMsg.nextInt(2)], MessageStore.CLASSIFICATION_DEFAULT);	
			if (uid == null) 
				System.out.println("null returned from addMessage() call to uid");
			else
				uriList.add(uid);
		}
		logger.info("total messages persisted to db: " + (uriList.size()-1));
		//loop back reading the messages from the db
		List<Message> messages = new ArrayList<Message>();
		for (int x=0; x<uriList.size(); x++) {
			messages.add(store.getMessage(uriList.get(x)));
		}
		logger.info("total messages read from db: " + (uriList.size()));
		logger.info("time to finish write/read for this client: " + (System.currentTimeMillis() - startTime) + " milliseconds.");
	}
	
    @BeforeClass
    public static void runBeforeAllTests()
    {
        try {
            File testResourceDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/src/test/resources/");
            System.out.println("Current dir=" + testResourceDir.getCanonicalPath());
            DOMConfigurator.configure(testResourceDir.getCanonicalPath() + "/log4j.xml");
            File buildDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/build/");
            File resourceDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/src/main/resources/");
            System.setProperty("org.jboss.soa.esb.propertyFile", "jbossesb-unittest-properties.xml");                    
            if ("org.hsqldb.jdbcDriver".equals(Configuration.getStoreDriver())) {
                final String databaseFile = buildDir + "/hsqltestdb" ;
                HsqldbUtil.dropHsqldb(databaseFile);
                server = HsqldbUtil.startHsqldb(databaseFile, "jbossesb");
                
                //Drop what is there now, if exists. We want to start fresh.                
                String sqlCreateCmd    = TestEnvironmentUtil.readTextFile(new File(resourceDir.getCanonicalPath() + "/message-store-sql/hsqldb/create_database.sql"));
                String sqlDropCmd      = TestEnvironmentUtil.readTextFile(new File(resourceDir.getAbsolutePath() + "/message-store-sql/hsqldb/drop_database.sql"));
                
                ConnectionManager mgr = ConnectionManagerFactory.getConnectionManager();
                mgr.init();
                Connection con = mgr.getConnection();
                Statement stmnt = con.createStatement();
                System.out.println("Dropping the schema if exist");
                stmnt.execute(sqlDropCmd);
                System.out.println("Creating the message store schema");
                stmnt.execute(sqlCreateCmd);
            }
        } catch (Throwable e) {
            e.printStackTrace();
            System.out.println("We should stop testing, since we don't have a db.");
            assertTrue(false);
        }
    }
	
	@AfterClass
	public static void runAfterAllTests() throws Exception{
        System.out.println("Stopping");
		if (Configuration.getStoreDriver().equals("org.hsqldb.jdbcDriver")) {
			HsqldbUtil.stopHsqldb(server);
		}
	}
	
	public static junit.framework.Test suite() {
		return new JUnit4TestAdapter(MessageStoreUnitTest.class);
	}
	
	
}
