/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.config.mappers110;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jms.Queue;
import javax.jms.Topic;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.Listener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JbossesbDocument.Factory;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JmsListenerDocument.JmsListener;
import org.jboss.soa.esb.listeners.jca.ActivationMapper;
import org.jboss.soa.esb.listeners.jca.JBossActivationMapper;
import org.jboss.soa.esb.listeners.jca.JcaConstants;
import org.junit.BeforeClass;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * Tests the JCA activation mapping of the JmsListenerMapper
 * 
 * @author <a href='mailto:kevin.conner@jboss.com'>Kevin Conner</a>
 */
public class JmsListenerMapperJCAActivationUnitTest
{
    private static final String TEST_DESTINATION = "TestDestination" ;
    private static final String TEST_PROVIDER_ADAPTER_JNDI = "TestProviderAdapterJNDI" ;
    private static final String TEST_DESTINATION_TYPE = "TestDestinationType" ;
    private static final String TEST_MESSAGE_SELECTOR = "TestMessageSelector" ;
    private static final String TEST_MAX_THREADS = "TestMaxThreads" ;

    private static final String TEST_QUEUE = "TestQueue" ;
    private static final String TEST_TOPIC = "TestTopic" ;
    private static final String DEFAULT_TEST_PROVIDER_ADAPTER_JNDI = "DefaultTestProviderAdapterJNDI" ;
    private static final String DEFAULT_TEST_MESSAGE_SELECTOR = "DefaultTestMessageSelector" ;
    private static final String DEFAULT_TEST_MAX_THREADS = "DefaultTestMaxThreads" ;
    
    @BeforeClass
    public static void setUp () throws Exception
    {
        final PropertyManager propertyManager = ModulePropertyManager.getPropertyManager(ModulePropertyManager.JCA_MODULE) ;
        propertyManager.setProperty(Environment.JCA_ACTIVATION_MAPPER + ".jms-ra.rar", JBossActivationMapper.class.getName()) ;
        propertyManager.setProperty(Environment.JCA_ACTIVATION_MAPPER + ".test-ra.rar", TestActivationMapper.class.getName()) ;
    }

    @Test
    public void testDefaultAllValues()
        throws Exception
    {
        final Map<String, String> activationConfig = generateConfig("jbossesb_jca_config_01.xml") ;
        
        final Map<String, String> expected = new HashMap<String, String>() ;
        expected.put("destination", "DestName") ;
        expected.put("providerAdapterJNDI", "ProviderAdapterJNDI") ;
        expected.put("destinationType", Queue.class.getName()) ;
        expected.put("messageSelector", "selector") ;
        expected.put("maxSession", "100") ;
        
        verifyConfiguration(expected, activationConfig) ;
    }

    @Test
    public void testDefaultMandatoryValues()
        throws Exception
    {
        final Map<String, String> activationConfig = generateConfig("jbossesb_jca_config_02.xml") ;
        
        final Map<String, String> expected = new HashMap<String, String>() ;
        expected.put("destination", "DestName") ;
        expected.put("destinationType", Topic.class.getName()) ;
        
        verifyConfiguration(expected, activationConfig) ;
    }

    @Test
    public void testAdapterAllValues()
        throws Exception
    {
        final Map<String, String> activationConfig = generateConfig("jbossesb_jca_config_03.xml") ;
        
        final Map<String, String> expected = new HashMap<String, String>() ;
        expected.put(TEST_DESTINATION, "DestName") ;
        expected.put(TEST_PROVIDER_ADAPTER_JNDI, "ProviderAdapterJNDI") ;
        expected.put(TEST_DESTINATION_TYPE, TEST_QUEUE) ;
        expected.put(TEST_MESSAGE_SELECTOR, "selector") ;
        expected.put(TEST_MAX_THREADS, "100") ;
        
        verifyConfiguration(expected, activationConfig) ;
    }

    @Test
    public void testAdapterMandatoryValues()
        throws Exception
    {
        final Map<String, String> activationConfig = generateConfig("jbossesb_jca_config_04.xml") ;
        
        final Map<String, String> expected = new HashMap<String, String>() ;
        expected.put(TEST_DESTINATION, "DestName") ;
        expected.put(TEST_PROVIDER_ADAPTER_JNDI, DEFAULT_TEST_PROVIDER_ADAPTER_JNDI) ;
        expected.put(TEST_DESTINATION_TYPE, TEST_TOPIC) ;
        expected.put(TEST_MESSAGE_SELECTOR, DEFAULT_TEST_MESSAGE_SELECTOR) ;
        expected.put(TEST_MAX_THREADS, DEFAULT_TEST_MAX_THREADS) ;
        
        verifyConfiguration(expected, activationConfig) ;
    }

    @Test
    public void testListenerPropertyAllValues()
        throws Exception
    {
        final Map<String, String> activationConfig = generateConfig("jbossesb_jca_config_05.xml") ;
        
        final Map<String, String> expected = new HashMap<String, String>() ;
        expected.put(TEST_DESTINATION, "DestName") ;
        expected.put(TEST_PROVIDER_ADAPTER_JNDI, "ProviderAdapterJNDI") ;
        expected.put(TEST_DESTINATION_TYPE, TEST_QUEUE) ;
        expected.put(TEST_MESSAGE_SELECTOR, "selector") ;
        expected.put(TEST_MAX_THREADS, "100") ;
        
        verifyConfiguration(expected, activationConfig) ;
    }

    @Test
    public void testListenerPropertyMandatoryValues()
        throws Exception
    {
        final Map<String, String> activationConfig = generateConfig("jbossesb_jca_config_06.xml") ;
        
        final Map<String, String> expected = new HashMap<String, String>() ;
        expected.put(TEST_DESTINATION, "DestName") ;
        expected.put(TEST_PROVIDER_ADAPTER_JNDI, DEFAULT_TEST_PROVIDER_ADAPTER_JNDI) ;
        expected.put(TEST_DESTINATION_TYPE, TEST_TOPIC) ;
        expected.put(TEST_MESSAGE_SELECTOR, DEFAULT_TEST_MESSAGE_SELECTOR) ;
        expected.put(TEST_MAX_THREADS, DEFAULT_TEST_MAX_THREADS) ;
        
        verifyConfiguration(expected, activationConfig) ;
    }

    @Test
    public void testBusPropertyAllValues()
        throws Exception
    {
        final Map<String, String> activationConfig = generateConfig("jbossesb_jca_config_07.xml") ;
        
        final Map<String, String> expected = new HashMap<String, String>() ;
        expected.put(TEST_DESTINATION, "DestName") ;
        expected.put(TEST_PROVIDER_ADAPTER_JNDI, "ProviderAdapterJNDI") ;
        expected.put(TEST_DESTINATION_TYPE, TEST_QUEUE) ;
        expected.put(TEST_MESSAGE_SELECTOR, "selector") ;
        expected.put(TEST_MAX_THREADS, "100") ;
        
        verifyConfiguration(expected, activationConfig) ;
    }

    @Test
    public void testBusPropertyMandatoryValues()
        throws Exception
    {
        final Map<String, String> activationConfig = generateConfig("jbossesb_jca_config_08.xml") ;
        
        final Map<String, String> expected = new HashMap<String, String>() ;
        expected.put(TEST_DESTINATION, "DestName") ;
        expected.put(TEST_PROVIDER_ADAPTER_JNDI, DEFAULT_TEST_PROVIDER_ADAPTER_JNDI) ;
        expected.put(TEST_DESTINATION_TYPE, TEST_TOPIC) ;
        expected.put(TEST_MESSAGE_SELECTOR, DEFAULT_TEST_MESSAGE_SELECTOR) ;
        expected.put(TEST_MAX_THREADS, DEFAULT_TEST_MAX_THREADS) ;
        
        verifyConfiguration(expected, activationConfig) ;
    }

    @Test
    public void testProviderPropertyAllValues()
        throws Exception
    {
        final Map<String, String> activationConfig = generateConfig("jbossesb_jca_config_09.xml") ;
        
        final Map<String, String> expected = new HashMap<String, String>() ;
        expected.put(TEST_DESTINATION, "DestName") ;
        expected.put(TEST_PROVIDER_ADAPTER_JNDI, "ProviderAdapterJNDI") ;
        expected.put(TEST_DESTINATION_TYPE, TEST_QUEUE) ;
        expected.put(TEST_MESSAGE_SELECTOR, "selector") ;
        expected.put(TEST_MAX_THREADS, "100") ;
        
        verifyConfiguration(expected, activationConfig) ;
    }

    @Test
    public void testProviderPropertyMandatoryValues()
        throws Exception
    {
        final Map<String, String> activationConfig = generateConfig("jbossesb_jca_config_10.xml") ;
        
        final Map<String, String> expected = new HashMap<String, String>() ;
        expected.put(TEST_DESTINATION, "DestName") ;
        expected.put(TEST_PROVIDER_ADAPTER_JNDI, DEFAULT_TEST_PROVIDER_ADAPTER_JNDI) ;
        expected.put(TEST_DESTINATION_TYPE, TEST_TOPIC) ;
        expected.put(TEST_MESSAGE_SELECTOR, DEFAULT_TEST_MESSAGE_SELECTOR) ;
        expected.put(TEST_MAX_THREADS, DEFAULT_TEST_MAX_THREADS) ;
        
        verifyConfiguration(expected, activationConfig) ;
    }

    private Map<String, String> generateConfig(final String config)
        throws Exception
    {
        final InputStream is = getClass().getResourceAsStream(config) ;
        assertNotNull("Configuration " + config, is) ;
        final XMLBeansModel model = new XMLBeansModel(Factory.parse(is).getJbossesb());
        
        final List<Listener> awareListeners = model.getESBAwareListeners() ;
        assertEquals("Listener count", 1, awareListeners.size()) ;
        
        final Listener listener = awareListeners.get(0) ;
        assertTrue("JmsListener", listener instanceof JmsListener) ;
        
        final Document doc = YADOMUtil.createDocument() ;
        final Element root = doc.createElement("root") ;
        final Element listenerElement = JmsListenerMapper.map(root, (JmsListener)listener, model) ;
        assertNotNull("Listener Element", listenerElement) ;
        
        final NodeList activationNodes = listenerElement.getElementsByTagName(JcaConstants.ELEMENT_ACTIVATION_CONFIG) ;
        final int numActivationNodes = (activationNodes == null ? 0 : activationNodes.getLength()) ;
        assertEquals("Activation node count", 1, numActivationNodes) ;
        
        final Element activationConfig = (Element)activationNodes.item(0) ;
        final NodeList propertyNodes = activationConfig.getElementsByTagName(JcaConstants.ELEMENT_PROPERTY) ;
        
        final Map<String, String> properties = new HashMap<String, String>() ;
        final int numProperties = (propertyNodes == null ? 0 : propertyNodes.getLength()) ;
        for(int count = 0 ; count < numProperties ; count++)
        {
            final Node node = propertyNodes.item(count) ;
            if (node instanceof Element)
            {
                final Element element = (Element)node ;
                final String name = element.getAttribute("name") ;
                final String value = element.getAttribute("value") ;
                properties.put(name, value) ;
            }
        }
        return properties ;
    }

    private static void verifyConfiguration(final Map<String, String> expected, final Map<String, String> activationConfig)
    {
        assertEquals("Number of activation properties", expected.size(), activationConfig.size()) ;
        
        for(Map.Entry<String, String> entry: expected.entrySet())
        {
            final String name = entry.getKey() ;
            assertEquals("Value of " + name, expected.get(name), activationConfig.get(name)) ;
        }
    }
    
    public static class TestActivationMapper implements ActivationMapper
    {
        public void setDestination(final Map<String, String> activationConfig, final String name)
            throws ConfigurationException
        {
            activationConfig.put(TEST_DESTINATION, name) ;
        }
        
        public void setProviderAdapterJNDI(final Map<String, String> activationConfig, final String providerAdapterJNDI)
            throws ConfigurationException
        {
            if (providerAdapterJNDI != null)
            {
                activationConfig.put(TEST_PROVIDER_ADAPTER_JNDI, providerAdapterJNDI) ;
            }
            else
            {
                activationConfig.put(TEST_PROVIDER_ADAPTER_JNDI, DEFAULT_TEST_PROVIDER_ADAPTER_JNDI) ;
            }
        }
        
        public void setDestinationType(final Map<String, String> activationConfig, final boolean queue)
            throws ConfigurationException
        {
            if (queue)
            {
                activationConfig.put(TEST_DESTINATION_TYPE, TEST_QUEUE) ;
            }
            else
            {
                activationConfig.put(TEST_DESTINATION_TYPE, TEST_TOPIC) ;
            }
        }
        
        public void setMessageSelector(final Map<String, String> activationConfig, final String messageSelector)
            throws ConfigurationException
        {
            if (messageSelector != null)
            {
                activationConfig.put(TEST_MESSAGE_SELECTOR, messageSelector) ;
            }
            else
            {
                activationConfig.put(TEST_MESSAGE_SELECTOR, DEFAULT_TEST_MESSAGE_SELECTOR) ;
            }
        }

        public void setMaxThreads(final Map<String, String> activationConfig, final Integer maxThreads)
            throws ConfigurationException
        {
            if (maxThreads != null)
            {
                activationConfig.put(TEST_MAX_THREADS, maxThreads.toString()) ;
            }
            else
            {
                activationConfig.put(TEST_MAX_THREADS, DEFAULT_TEST_MAX_THREADS) ;
            }
        }
    }
    
    public static junit.framework.Test suite ()
    {
        return new JUnit4TestAdapter(JmsListenerMapperJCAActivationUnitTest.class);
    }
}
