/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.http;

import static org.junit.Assert.*;

import java.util.Properties;

import junit.framework.JUnit4TestAdapter;

import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.params.HttpConnectionManagerParams;
import org.jboss.soa.esb.http.configurators.Connection;
import org.junit.Test;

/**
 * Unit test for {@link HttpClientFactory}.
 * <p/>
 * 
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public class HttpClientFactoryUnitTest
{
    @Test
    public void testMaximumTotalConnections()
        throws Exception
    {
        final int maxTotalConnections = 1234 ;
        final Properties properties = getProperties(Integer.toString(maxTotalConnections), null) ;
        final HttpClient httpClient = HttpClientFactory.createHttpClient(properties) ;
        try
        {
            final HostConfiguration configuration = httpClient.getHostConfiguration() ;
            final HttpConnectionManagerParams params = httpClient.getHttpConnectionManager().getParams() ;
            assertEquals("MAX_TOTAL_CONNECTIONS not set", maxTotalConnections, params.getMaxTotalConnections()) ;
            assertTrue("maxConnectionsPerHost identical", maxTotalConnections !=  params.getMaxConnectionsPerHost(configuration)) ;
        }
        finally
        {
            HttpClientFactory.shutdown(httpClient) ;
        }
    }
    
    @Test
    public void testMaximumConnectionsPerHost()
        throws Exception
    {
        final int maxConnectionsPerHost = 1234 ;
        final Properties properties = getProperties(null, Integer.toString(maxConnectionsPerHost)) ;
        
        final HttpClient httpClient = HttpClientFactory.createHttpClient(properties) ;
        try
        {
            final HostConfiguration configuration = httpClient.getHostConfiguration() ;
            final HttpConnectionManagerParams params = httpClient.getHttpConnectionManager().getParams() ;
            assertEquals("MAX_CONNECTIONS_PER_HOST not set", maxConnectionsPerHost, params.getMaxConnectionsPerHost(configuration)) ;
            assertTrue("maxTotalConnections identical", maxConnectionsPerHost !=  params.getMaxTotalConnections()) ;
        }
        finally
        {
            HttpClientFactory.shutdown(httpClient) ;
        }
    }

    
    @Test
    public void testMultipleConfigurations()
        throws Exception
    {
        final int firstMaxTotalConnections = 1234 ;
        final int firstMaxConnectionsPerHost = 2345 ;
        final Properties firstProperties = getProperties(Integer.toString(firstMaxTotalConnections), Integer.toString(firstMaxConnectionsPerHost)) ;

        final int secondMaxTotalConnections = 3456 ;
        final int secondMaxConnectionsPerHost = 4567 ;
        final Properties secondProperties = getProperties(Integer.toString(secondMaxTotalConnections), Integer.toString(secondMaxConnectionsPerHost)) ;
        
        final HttpClient firstHttpClient = HttpClientFactory.createHttpClient(firstProperties) ;
        try
        {
            final HttpClient secondHttpClient = HttpClientFactory.createHttpClient(secondProperties) ;
            try
            {
                final HostConfiguration firstConfiguration = firstHttpClient.getHostConfiguration() ;
                final HttpConnectionManagerParams firstParams = firstHttpClient.getHttpConnectionManager().getParams() ;
                final HttpConnectionManager firstConnectionManager = firstHttpClient.getHttpConnectionManager() ;
                
                assertEquals("MAX_TOTAL_CONNECTIONS not set", firstMaxTotalConnections, firstParams.getMaxTotalConnections()) ;
                assertEquals("MAX_CONNECTIONS_PER_HOST not set", firstMaxConnectionsPerHost, firstParams.getMaxConnectionsPerHost(firstConfiguration)) ;
                assertNotNull("connection manager is null", firstConnectionManager) ;
                assertEquals("connection manager is not correct type", ESBMultiThreadedHttpConnectionManager.class, firstConnectionManager.getClass()) ;
                assertNotNull("HostConfiguration is null", ((ESBMultiThreadedHttpConnectionManager)firstConnectionManager).getHostConfiguration()) ;
                
                final HostConfiguration secondConfiguration = secondHttpClient.getHostConfiguration() ;
                final HttpConnectionManagerParams secondParams = secondHttpClient.getHttpConnectionManager().getParams() ;
                final HttpConnectionManager secondConnectionManager = secondHttpClient.getHttpConnectionManager() ;
                
                assertEquals("MAX_TOTAL_CONNECTIONS not set", secondMaxTotalConnections, secondParams.getMaxTotalConnections()) ;
                assertEquals("MAX_CONNECTIONS_PER_HOST not set", secondMaxConnectionsPerHost, secondParams.getMaxConnectionsPerHost(secondConfiguration)) ;
                assertNotNull("connection manager is null", secondConnectionManager) ;
                assertEquals("connection manager is not correct type", ESBMultiThreadedHttpConnectionManager.class, secondConnectionManager.getClass()) ;
                assertNotNull("HostConfiguration is null", ((ESBMultiThreadedHttpConnectionManager)secondConnectionManager).getHostConfiguration()) ;
            }
            finally
            {
                HttpClientFactory.shutdown(secondHttpClient) ;
            }
        }
        finally
        {
            HttpClientFactory.shutdown(firstHttpClient) ;
        }
    }

    private Properties getProperties(final String maxTotalConnections, final String maxConnectionsPerHost)
    {
        final Properties properties = new Properties() ;
        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, "http://example.com/targetHostUrl") ;
        
        if (maxTotalConnections != null)
        {
            properties.setProperty(Connection.MAX_TOTAL_CONNECTIONS, maxTotalConnections) ;
        }
        if (maxConnectionsPerHost != null)
        {
            properties.setProperty(Connection.MAX_CONNECTIONS_PER_HOST, maxConnectionsPerHost) ;
        }
        return properties ;
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(HttpClientFactoryUnitTest.class);
    }
}
