/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * (C) 2005-2006
 *  */
package org.jboss.soa.esb.actions.naming;

import java.net.URI;
import java.net.URISyntaxException;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.addressing.eprs.HTTPEpr;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;

import junit.framework.TestCase;

/**
 * Test the file name generator action.    This class contains tests for the
 * getters and setters, tests processing a message, and tests creating strategies
 * from valid and invalid file naming strategy XML.
 * 
 * @author tcunning@redhat.com
 */
public class FileNameGeneratorActionUnitTest extends TestCase {
	private static final Logger logger = Logger.getLogger(FileNameGeneratorActionUnitTest.class);
	private static final String FILENAME = "word";
	private static final String SUFFIX = "doc";
	private static final String NEWSUFFIX = "esbout";
	private static final String FILENAME_PROPERTY = "filename_property";
	private static final String RESULT_PROPERTY = "result_property";

	public void testGetters() {
		FileNameGeneratorAction fnga = new FileNameGeneratorAction();
		String test = new String("test");
		
		fnga.setResultProperty(test);
		assertEquals(test, fnga.getResultProperty());
		
		fnga.setFileNameProperty(test);
		assertEquals(test, fnga.getFileNameProperty());
	}
	
	public void testStrategies() {
		// Test a valid FileNamingStrategy
		String strategies = "<xml><strategy class=\"org.jboss.soa.esb.actions.naming.strategy.ChangeSuffixNamingStrategy\">blah</strategy></xml>";
		FileNameGeneratorAction fnga = new FileNameGeneratorAction();
		fnga.setStrategies(strategies);
		fnga.setFileNameProperty(FILENAME_PROPERTY);
		fnga.setResultProperty(RESULT_PROPERTY);
		logger.debug("FileNameStrategy : [" + fnga.getStrategies() + "]");
		
		// Test processing a message
		Message msg1 = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
		msg1.getBody().add("bar");
		try {
			msg1.getHeader().getCall().setTo(new HTTPEpr("http://foo.bar"));
			msg1.getHeader().getCall().setMessageID(new URI("urn:1234"));
			msg1.getProperties().setProperty(FILENAME_PROPERTY, FILENAME + "." + SUFFIX);
			@SuppressWarnings("unused")
			Message result = fnga.process(msg1);
			String filename = (String) result.getProperties().getProperty(RESULT_PROPERTY);
			logger.debug("Filename=[" + filename + "]");
			assertEquals(filename, FILENAME + "." + NEWSUFFIX);
		} catch (URISyntaxException e1) {
			logger.error("", e1);
			fail(e1.getMessage());
		} catch (ActionProcessingException ape) {
			logger.error("", ape);
			fail(ape.getMessage());
		} catch (Exception e) {
			logger.error("", e);
			fail(e.getMessage());
		}
		
		
		// Test an invalid FileNamingStrategy class
		strategies = "<xml><strategy class=\"foobar\">blah</strategy></xml>";
		try {
			fnga = new FileNameGeneratorAction();
			fnga.setStrategies(strategies);
			fail("Setting the strategy on an invalid class should result in an Exception.");
		} catch (RuntimeException re) {
		}
		logger.debug("FileNameStrategy : [" + fnga.getStrategies() + "]");
	}
}
