/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.security;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.net.URL;
import java.security.Principal;
import java.util.HashSet;
import java.util.Set;

import javax.security.auth.Subject;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.services.security.SecurityConfig;
import org.jboss.soa.esb.services.security.SecurityContext;
import org.jboss.soa.esb.services.security.SecurityService;
import org.jboss.soa.esb.services.security.SecurityServiceException;
import org.jboss.soa.esb.services.security.TestPrincipal;
import org.jboss.soa.esb.services.security.SecurityConfig.Builder;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.jboss.soa.esb.services.security.principals.Group;
import org.jboss.soa.esb.services.security.principals.Role;
import org.jboss.soa.esb.util.ClassUtil;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

/**
 * Unit test for {@link SecurityServiceImpl}
 * <p/>
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 * @since 4.4
 */
public class JaasSecurityServiceUnitTest
{
	private final SecurityService service = new JaasSecurityService();
	private final Subject subject = new Subject();
	private String jbossEsbProperties;

	@Test
	public void authenticateUserpassword() throws ConfigurationException, SecurityServiceException
	{
		final String userName = "testUser";
		final String password = "testPassword";
		Builder builder = new SecurityConfig.Builder("UserPassLogin");
		builder.callBackhandler("org.jboss.internal.soa.esb.services.security.UserPassCallbackHandler");
		SecurityConfig configInfo = builder.build();
		TestPrincipal principal = new TestPrincipal(userName);
		Set<Object> credentials = new HashSet<Object>();
		credentials.add(password);

		AuthenticationRequest authRequest = new AuthenticationRequestImpl.Builder(principal, credentials).build();
		SecurityContext context = new SecurityContext(subject, SecurityContext.getConfigurationTimeout());
		service.configure();
		service.authenticate(configInfo, context, authRequest);

		Set<TestPrincipal> principals = subject.getPrincipals( TestPrincipal.class );
		assertEquals( 1, principals.size() );
		assertEquals( userName, principals.iterator().next().getName() );
	}

	@Test
	public void authenticateWithRole() throws ConfigurationException, SecurityServiceException
	{
		Builder builder = new SecurityConfig.Builder("SuccessfulLogin");
		builder.runAs("adminRole");
		SecurityConfig configInfo = builder.build();
		SecurityContext context = new SecurityContext(subject, SecurityContext.getConfigurationTimeout());
		service.authenticate(configInfo, context, null);
		Set<Principal> principals = subject.getPrincipals( Principal.class );
		assertEquals( 2, principals.size() );
	}

	@Test
	public void authenticateWithExistingRole() throws ConfigurationException, SecurityServiceException
	{
		Builder builder = new SecurityConfig.Builder("SuccessfulLogin");
		builder.runAs("adminRole");
		SecurityConfig configInfo = builder.build();
		Group group = new Group("Roles");
		group.addMember(new Role("adminRole1"));

		subject.getPrincipals().add(group);

		SecurityContext context = new SecurityContext(subject, SecurityContext.getConfigurationTimeout());
		service.authenticate(configInfo, context, null);

		Set<Principal> principals = subject.getPrincipals( Principal.class );
		assertEquals( 2, principals.size() );
	}

	@Test
	public void isCallerInRole() throws SecurityServiceException
	{
		final String roleName = "adminRole";
		Builder builder = new SecurityConfig.Builder("SuccessfulLogin");
		builder.runAs(roleName);
		SecurityConfig configInfo = builder.build();
		SecurityContext context = new SecurityContext(subject, SecurityContext.getConfigurationTimeout());
		service.authenticate(configInfo, context, null);
		assertTrue( service.isCallerInRole(subject, new Role(roleName)));
	}

	@Test ( expected = SecurityServiceException.class )
	public void loginFailure() throws ConfigurationException, SecurityServiceException
	{
		Builder builder = new SecurityConfig.Builder("FailureLogin");
		SecurityConfig configInfo = builder.build();
		SecurityContext context = new SecurityContext(subject, SecurityContext.getConfigurationTimeout());
		service.authenticate( configInfo, context, null );
	}

	@Test
    public void rolesAllowed() throws ConfigurationException, SecurityServiceException
    {
        Builder builder = new SecurityConfig.Builder("SuccessfulLogin");
        builder.runAs("esbRole");
        builder.rolesAllowed("esbRole");
        SecurityConfig configInfo = builder.build();
		SecurityContext context = new SecurityContext(subject, SecurityContext.getConfigurationTimeout());
        service.authenticate( configInfo, context, null );
        boolean checkRolesAllowed = service.checkRolesAllowed(configInfo.getRolesAllowed(), context);
        assertTrue(checkRolesAllowed);
    }

	@Test
	public void refresh() throws SecurityServiceException
	{
		Runnable runnable = new Runnable()
		{
			public void run()
			{
				try
                {
                    threadLogin();
                }
				catch (final SecurityServiceException ignore)
                {
				    ignore.printStackTrace();
                }
			}
		};
		for (int i = 0; i < 1000; i++)
		{
    		Thread t1 = new Thread(runnable);
    		Thread t2 = new Thread(runnable);
    		t1.start();
    		t2.start();
		}
	}

	private void threadLogin() throws SecurityServiceException
	{
		Builder builder = new SecurityConfig.Builder("SuccessfulLogin");
		SecurityConfig configInfo = builder.build();
		SecurityContext context = new SecurityContext(new Subject(), SecurityContext.getConfigurationTimeout());
		try
		{
			service.authenticate( configInfo, context, null );
		}
		catch (SecurityServiceException e)
		{
    		fail(e.getMessage());
		}
	}

	@Before
	public void setup() throws ConfigurationException
	{
        jbossEsbProperties = System.getProperty(Environment.PROPERTIES_FILE);
		URL resource = ClassUtil.getResource("security-properties.xml", getClass());
        System.setProperty(Environment.PROPERTIES_FILE, "abs://" + resource.getFile());

		URL loginConfig = ClassUtil.getResource("test_jaas.config", getClass());
		System.setProperty("java.security.auth.login.config", loginConfig.getFile());
	}

	@After
	public void tearDown()
	{
		if ( jbossEsbProperties != null )
            System.setProperty(Environment.PROPERTIES_FILE, jbossEsbProperties);
	}

	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( JaasSecurityServiceUnitTest.class );
	}

}
