/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.util;

import java.io.File;
import java.io.IOException;

public interface RemoteFileSystem
{

        public static final String FTP_PROTOCOL = "ftp";

        public static final String SFTP_PROTOCOL = "sftp";

        public static final String FTPS_PROTOCOL = "ftps";

        public static final String PARMS_PROTOCOL = "protocol";

        public static final String PARMS_FTP_SERVER = "ftpServer";

        public static final String PARMS_USER = "ftpUser";

        public static final String PARMS_PASSWD = "ftpPassword";

        public static final String PARMS_PORT = "ftpPort";

        public static final String PARMS_REMOTE_DIR = "ftpRemoteDir";

        public static final String PARMS_LOCAL_DIR = "ftpLocalDir";

        public static final String PARMS_ASCII = "ftpAscii";

        public static final String PARMS_PASSIVE = "ftpPassive";

        public static final String PARMS_CERTIFICATE = "certificate";

        public static final String PARMS_CERTIFICATE_NAME = "certificateName";

        public static final String PARMS_PASSPHRASE = "passphrase";
        
        public static final String PARMS_RENAME_RETRY = "ftpRenameRetry";
        
        public static final String PARMS_TIMEOUT_DEFAULT = "ftpTimeoutDefault";
        
        public static final String PARMS_TIMEOUT_DATA = "ftpTimeoutData";
        
        public static final String PARMS_TIMEOUT_SO = "ftpTimeoutSo";
        
        public static final String PARMS_CONTROL_CHANNEL_ENCODING = "ftpControlChannelEncoding";
        
        public String getRemoteDir ();

        /**
         * Terminates ftp session and frees resources
         * <li>Well behaved programs should make sure to call this method </li>
         */
        public void quit (); // _________________________________

        /**
         * Deletes specified file in remote directory
         * 
         * @param p_sFile
         *            String : filename to delete. Method will attempt to delete
         *            file with rightmost node of argument within remote directory
         *            specified in 'remoteDirURI'
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or file cannot be
         *             deleted in remote directory
         */
        public void deleteRemoteFile (String p_sFile) throws RemoteFileSystemException; // _________________________________

        public void remoteDelete (File p_oFile) throws RemoteFileSystemException; // _________________________________

        /**
         * Gets the list of files in the remote directory that end with arg0
         * 
         * @param p_sSuffix
         *            String : retrieve only files that end with that suffix - all
         *            files if null
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or problems
         *             encountered
         */
        public String[] getFileListFromRemoteDir (String p_sSuffix)
                        throws IOException, RemoteFileSystemException; // _________________________________

        /**
         * Change remote directory
         * 
         * @param p_sDir
         *            String : directory to set
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or problems
         *             encountered
         */
        public void setRemoteDir (String p_sDir) throws RemoteFileSystemException; // _________________________________

        /**
         * Renames specified file in remote directory to specified new name
         * 
         * @param p_sFrom
         *            String : filename to rename
         * @param p_sTo
         *            String : new filename
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or file cannot be
         *             renamed to new name in remote directory
         *             <li>Method will attempt to rename file with rightmost node
         *             of argument within remote directory specified in
         *             'remoteDirURI', to new name inside the SAME remote directory
         */
        public void renameInRemoteDir (String p_sFrom, String p_sTo)
                        throws RemoteFileSystemException; // _________________________________

        /**
         * 
         * @param p_oFrom
         * @param p_oTo
         * @throws IOException if file not found.
         * @throws RemoteFileSystemException
         */
        public void remoteRename (File p_oFrom, File p_oTo) throws IOException, RemoteFileSystemException; // _________________________________

        /**
         * Uploads specified file from local directory (localDirURI) to remote
         * directory (remoteDirURI)
         * 
         * @param p_oFile
         *            String : filename to upload
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or file cannot be
         *             uploaded
         *             <li> local file will be renamed during transfer
         *             ('.xferNotReady' appended to name)</li>
         *             <li> upon successful completion. the suffix '.xferDone' will
         *             be appended to the original filename </li>
         */
        public void uploadFile (File p_oFile, String p_sRemoteName)
                        throws RemoteFileSystemException; // _________________________________

        /**
         * Downloads specified file from remote directory (remoteDirURI) to local
         * directory (localDirURI)
         * 
         * @param p_sFile
         *            String : filename to download
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or file cannot be
         *             downloaded
         *             <li> local file is assigned a temporary name during transfer
         *             </li>
         *             <li> upon successful completion, local temporary file will be
         *             renamed to name specified in argument, and suffix '.xferDone'
         *             will be appended to the original filename in the remote
         *             directory </li>
         */
        public void downloadFile (String p_sFile, String p_sFinalName)
                        throws IOException, RemoteFileSystemException; // _________________________________

}