/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.notification;

import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.jms.DeliveryMode;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.naming.Context;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.rosetta.pooling.ConnectionException;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPoolContainer;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.util.JndiUtil;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.notification.jms.DefaultJMSPropertiesSetter;
import org.jboss.soa.esb.notification.jms.JMSPropertiesSetter;

/**
 * Abstract class that defines the behaviour of NotifyQueues and NotifyTopics
 * (and in fact any other javax.jms.Destination object), and provides common
 * methods to derived classes
 * <p>
 * Description:
 * </p>
 * <p>
 * Author: Heuristica - Buenos Aires - Argentina
 * </p>
 * 
 * @version 1.0
 */
public abstract class NotifyJMS extends NotificationTarget
{
    /*
     * Value can be used to determine how the notification message
     * will be represented on the JMS "wire".
     */
    
    public static enum NativeMessage { text, object }
    
    protected Logger log = Logger.getLogger(this.getClass());

	/**
	 * Connection Factory JNDI name.
	 */
	public static final String CONNECTION_FACTORY = "ConnectionFactory";
    
    /**
     * Strategy for setting JMSProperties
     */
    private JMSPropertiesSetter jmsPropertiesStrategy = new DefaultJMSPropertiesSetter();
    private MessagePayloadProxy payloadProxy;

    /**
	 * Abstract method - All classes that extend NotifyJMS must implement it
	 * according to their own javax.jms.Destination needs (Queue/Topic)
	 * 
	 * @param p_oMsg			the JMS Message to send
	 * @param msgProducer		the JMS Message Producer which will perform the sending
	 * @throws JMSException 	if the send was not sucessful
	 */
	protected abstract void send(final Message p_oMsg, MessageProducer msgProducer ) throws JMSException;
	
	/**
	 * Creates a message producer specific to the subclasses implementation
	 * i.e QueueSender or TopicPublisher.
	 * 
	 * @param pool
	 * @param sAtt
	 * @param session
	 * @param environment
	 * @return MessageProducer
	 * @throws NamingException
	 * @throws JMSException
	 * @throws ConnectionException
	 */
	protected abstract MessageProducer createProducer( final JmsConnectionPool pool, 
			final String destinationName, 
			final Session session,
			final Properties environment) throws NamingException, JMSException, ConnectionException;

	/**
	 * Element name mnemonic to search for child elements in the ConfigTree at
	 * constructor time, that will hold a "name" "value" pair to be included in
	 * the JMS message header
	 */
	public static final String CHILD_MSG_PROP = "messageProp";

	/**
	 * Attribute name mnemonic for the name of the message property to fill with
	 * contents of the "value" aattribute
	 */
	public static final String ATT_PROP_NAME = "name";

	/**
	 * Attribute name mnemonic for the value to assign to the corresponding
	 * message property
	 */
	public static final String ATT_PROP_VALUE = "value";

	/**
	 * Attribute name mnemonic for the att that will hold the JNDI name of the
	 * destination javax.jms.Destination object
	 */
	public static final String ATT_DEST_NAME = "jndiName";

	/**
	 * Constant used in configuration 
	 */
	public static final String PERSISTENT_ATTR = "persistent";

	/**
	 * Constant used in configuration
	 */
	public static final String PRIORITY_ATTR = "priority";

	/**
	 * Constant used in configuration
	 */
	public static final String TIME_TO_LIVE_ATTR = "time-to-live";
	
	private Properties m_oProps = new Properties();

	/**
	 * The javax.jms.Connection instance used to talk to JMS
	 */
	protected JmsConnectionPool[] connectionPools;

	/**
	 * The javax.jms.Session instance used to talk to JMS
	 */
	protected Session[] sessions;

	/**
	 * Array with an instance of javax.jms.MessageProducer on each entry that
	 * will be used to send the notification at sendNotification() time
	 */
	protected MessageProducer[] producers;
	
	/**
	 * Delivery mode for JMS Messages, either DeliveryMode.PERSISTENT
	 * or DeliveryMode.NON_PERSISTENT
	 */
	protected int[] deliveryModes;
	
	/**
	 * Priorities for JMS Messages seen by this notifier
	 */
	protected int[] priorities;
	
	/**
	 * Time-to-lives for JMS Messages sen by this notifier
	 */
	protected long[] timeToLives;

	private Properties environment;

	/**
	 * Constructor that will be called by child classes to perform
	 * initialization tasks that are common to all derived classes
	 * 
	 * @param p_oP
	 *            ConfigTree
	 *            <p>
	 *            Will contain a variable number of "messageProp" child elements
	 *            that will be added to messages sent to
	 *            </p>
	 * @throws ConfigurationException
	 */
	protected NotifyJMS (ConfigTree p_oP) throws ConfigurationException
	{
		super(p_oP);

		ConfigTree[] oaMsgP = p_oP.getChildren(CHILD_MSG_PROP);
		for (int i1 = 0; i1 < oaMsgP.length; i1++)
		{
			String sKey = oaMsgP[i1].getAttribute(ATT_PROP_NAME);
			if (null == sKey) continue;
			String sVal = oaMsgP[i1].getAttribute(ATT_PROP_VALUE);
			if (null == sVal) continue;

			m_oProps.setProperty(sKey.trim(), sVal);
		}
        payloadProxy = new MessagePayloadProxy(p_oP,
                                               new String[] {BytesBody.BYTES_LOCATION},
                                               new String[] {BytesBody.BYTES_LOCATION});
        payloadProxy.setNullGetPayloadHandling(MessagePayloadProxy.NullPayloadHandling.LOG);
    } // __________________________________

	/**
	 * Send a JMS message using p_o to fill in the message content and the list
	 * of message properties that will be added to the JMS message header fields
	 * 
	 * @param esbMessage
	 *            Object - This object or thie object's toString() method will
	 *            supply contents of JMS message
	 * @see NotifyJMS#CHILD_MSG_PROP
	 */
	public void sendNotification (org.jboss.soa.esb.message.Message esbMessage) throws NotificationException
	{
	    /*
	     * Blah. We do this for backward compatibility.
	     */
	    
	    NativeMessage defaultType;
	    
		if (MessageType.JAVA_SERIALIZED.equals(esbMessage.getType()))
		    defaultType = NativeMessage.object;
		else
		    defaultType = NativeMessage.text;
	    
	    /*
	     * http://jira.jboss.com/jira/browse/JBESB-1606
	     * 
	     * Prior to this, the JMS message type transmitted depended on the
	     * ESB message type! That should be opaque and not affect how the contents
	     * are transmitted. So we fixed it, but need to provide backward
	     * compatibility for now. But the old approach/"feature" is essentially
	     * deprecated.
	     * 
	     * http://jira.jboss.com/jira/browse/JBESB-1607
	     */
	    
	    try
	    {
		Message jmsMessage = null;
		Object obj = payloadProxy.getPayload(esbMessage);
		NativeMessage type = (NativeMessage) esbMessage.getProperties().getProperty(Environment.JMS_NATIVE_MESSAGE_TYPE, defaultType);
		
		if (type == NativeMessage.object)
		{
		    if(obj == null) {
			// create a null payload message...
			jmsMessage = sessions[0].createObjectMessage(null);
		    } else if(obj instanceof byte[]) {
			jmsMessage = sessions[0].createObjectMessage((byte[]) obj);
		    } else {
			throw new NotificationException("Expected payload type for '" + esbMessage.getType() + "' is byte[].  Received '" + obj.getClass().getName() + "'.");
		    }
		}
		else
		{
		    String content = null;
		    if (obj instanceof byte[]) {
			content = new String((byte[]) obj);
		    } else if(obj != null) {
			content = obj.toString();
		    }
		    jmsMessage = sessions[0].createTextMessage(content);
		}
		
		setJMSProperties( esbMessage, jmsMessage );
		setJMSReplyTo( jmsMessage, esbMessage );

		for (Iterator<Object> II = m_oProps.keySet().iterator(); II.hasNext();)
		{
		    String sKey = (String) II.next();
		    String sVal = m_oProps.getProperty(sKey);
		    jmsMessage.setStringProperty(sKey, sVal);
		}
		sendToAll(jmsMessage);
	    }
	    catch (JMSException ex)
	    {
		throw new NotificationException(ex);
	    } catch (MessageDeliverException e) {
		throw new NotificationException(e);
	    }
            finally
            {
        	/*
        	 * Why do we create multiple pools but only ever use one?
        	 * 
        	 * Don't forget to put the session back into the pool!
        	 */
        	
        	if (connectionPools != null)
        	    connectionPools[0].closeSession(sessions[0]);
            }
    } // __________________________________
	
	/**
	 * This method will set appropriate JMSProperties on the outgoing JMSMessage object.
	 * </p>
	 * Sublclasses can either override this method add a different behaviour, or they can 
	 * set the strategy by calling {@link #setJmsPropertiesStrategy(JMSPropertiesSetter)}.
	 * </p> 
	 * See {@link org.jboss.soa.esb.notification.jms.JMSPropertiesSetter} for more info.
	 */
	protected void setJMSProperties(org.jboss.soa.esb.message.Message fromESBMessage, Message toJMSMessage ) throws JMSException { 	
		jmsPropertiesStrategy.setJMSProperties( fromESBMessage, toJMSMessage );
	}
	
	void setJMSReplyTo( final Message jmsMessage, final org.jboss.soa.esb.message.Message esbMessage ) throws NotificationException
	{
		Destination destination = (Destination) esbMessage.getProperties().getProperty( JMSPropertiesSetter.JMS_REPLY_TO );
		if ( destination != null )
		{
			try
			{
    			jmsMessage.setJMSReplyTo( (Destination) destination );
			}
			catch (JMSException e)
			{
				throw new NotificationException( "Could not set the JMSReplyTo to destination [" + destination + "]", e );
			} 
		}
	}
	
	/**
	 * Template method for sending JMS messages to destinations.
	 * </p>
	 * Subclasses implement send(Message, MessageProducer) which 
	 * performs sending/publishing.
	 * 
	 * @param p_oMsg			the JMS Message to send.
	 * @throws JMSException 	if any of the sends fail a JMSException will be
	 * 							thrown. The exception will contain an error
	 * 							message and stacktrace for every failed send. 
	 */
	protected void sendToAll (final Message p_oMsg) throws JMSException
	{
            final StringBuilder jmsExceptions = new StringBuilder();
			for (int i1 = 0; i1 < producers.length; i1++)
			{
				try
				{
					send( p_oMsg, producers[i1]);
				}
				catch (final JMSException e)
				{
					final String msg = "[JMSException while sending to : " + producers[i1].getDestination();
	                log.error(msg, e);
	                jmsExceptions.append( NotifyUtil.createExceptionErrorString( msg, e ));
				}
			}
			if ( jmsExceptions.length() > 0 )
				throw new JMSException( jmsExceptions.toString() );
	}
	
	/**
	 * Will setup/create JMS connections, sessions, producers.
	 * 
	 * @param configTrees
	 * @throws ConfigurationException
	 * @throws JMSException
	 * @throws ConnectionException
	 */
	protected void setUpProducers (final ConfigTree[] configTrees) throws ConfigurationException, JMSException, ConnectionException
	{
		// REVIEW: The connection factory name is hardcoded and is the same as
		// that of the queue connection factory.
		final int nrQueuesOrTopics = configTrees.length;
		producers = new MessageProducer[nrQueuesOrTopics];
        connectionPools = new JmsConnectionPool[nrQueuesOrTopics];
        sessions = new Session[nrQueuesOrTopics];
        deliveryModes = new int[nrQueuesOrTopics];
        priorities = new int[nrQueuesOrTopics];
        timeToLives = new long[nrQueuesOrTopics];
        
		try
		{
			for (int i = 0; i < nrQueuesOrTopics; i++)
			{
                environment = new Properties();
				String sAtt = configTrees[i].getAttribute(ATT_DEST_NAME);
				if (null == sAtt) throw new ConfigurationException("Missing attribute '" + ATT_DEST_NAME);
				
                String jndiURL = configTrees[i].getAttribute(JMSEpr.JNDI_URL_TAG);
                if (jndiURL!=null) environment.setProperty(Context.PROVIDER_URL, jndiURL);
                
                String contextFactory = configTrees[i].getAttribute(JMSEpr.JNDI_CONTEXT_FACTORY_TAG);
                if (contextFactory!=null) environment.setProperty(Context.INITIAL_CONTEXT_FACTORY, contextFactory);
                
                String prefix = configTrees[i].getAttribute(JMSEpr.JNDI_PKG_PREFIX_TAG);
                if (prefix!=null) environment.setProperty(Context.URL_PKG_PREFIXES, prefix);
                List<KeyValuePair> properties=configTrees[i].childPropertyList();
                final String[] jndiPrefixes = JndiUtil.getJndiPrefixes(KeyValuePair.getValue(JMSEpr.JNDI_PREFIXES, properties)) ;
                for (KeyValuePair property : properties) {
                    for(String jndiPrefix: jndiPrefixes) {
                        if (property.getKey().startsWith(jndiPrefix)) {
                            environment.setProperty(property.getKey(), property.getValue());
                            break ;
                        }
                    }
                }
                
                String connectionFactory = configTrees[i].getAttribute(JMSEpr.CONNECTION_FACTORY_TAG, CONNECTION_FACTORY);
                
                connectionPools[i] = JmsConnectionPoolContainer.getPool(environment, connectionFactory);
                sessions[i] = connectionPools[i].getSession() ;
                producers[i] = createProducer( connectionPools[i], sAtt, sessions[i], environment );
                
                final String persistentStr = configTrees[i].getAttribute( PERSISTENT_ATTR, "true" );
                deliveryModes[i] = persistentStr.equalsIgnoreCase( "true" ) ? DeliveryMode.PERSISTENT : DeliveryMode.NON_PERSISTENT;
                
                final String priorityStr = configTrees[i].getAttribute( PRIORITY_ATTR );
                priorities[i] = priorityStr == null ? Message.DEFAULT_PRIORITY : Integer.parseInt( priorityStr );
                
                final String ttlStr = configTrees[i].getAttribute( TIME_TO_LIVE_ATTR );
                timeToLives[i] = ttlStr == null ? Message.DEFAULT_TIME_TO_LIVE : Long.parseLong( ttlStr );
                
				producers[i].setDeliveryMode( deliveryModes[i] );
				producers[i].setPriority( priorities[i] );
				producers[i].setTimeToLive( timeToLives[i] );
			}
		}
		catch (NamingException ex)
		{
			throw new ConfigurationException(ex);
		}
	}

	/**
	 * Sets the strategy for handling the setting of properties on an outgoing 
	 * JMS Message.
	 * 
	 * @param jmsPropertiesStrategy the strategy to use.
	 */
	public void setJmsPropertiesStrategy( JMSPropertiesSetter jmsPropertiesStrategy )
	{
		this.jmsPropertiesStrategy = jmsPropertiesStrategy;
	}
	
} // ____________________________________________________________________________
