/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.config.mappers120;

import org.jboss.soa.esb.schedule.*;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycle;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.ScheduleProviderDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.SimpleScheduleDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.CronScheduleDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.FrequencyUnit;

import java.util.List;
import java.util.Properties;
import java.util.ArrayList;

/**
 * Factory class for creating and configuring a {@link org.jboss.soa.esb.schedule.ScheduleProvider}
 * instance.
 *
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class ScheduleProviderFactory {

    public static final String ATTR_SCHEDULE_ID_REF = "scheduleidref";
    public static final String ATTR_FREQUENCY = "schedule-frequency";

    public static ScheduleProvider createInstance(List<ManagedLifecycle> listeners, ScheduleProviderDocument.ScheduleProvider scheduleProviderConfig) throws ConfigurationException {
        Properties schedulerConfig = null;
        List<Schedule> schedules = null;
        ScheduleProvider instance;

        if(scheduleProviderConfig != null) {
            schedulerConfig = XMLBeansModel.toProperties(scheduleProviderConfig.getPropertyList());
            schedules = getSchedules(scheduleProviderConfig);
        }
        instance = new ScheduleProvider(schedulerConfig, schedules);

        try {
            for(ManagedLifecycle listener : listeners) {
                if(listener instanceof ScheduledEventListener) {
                    ConfigTree config = listener.getConfig();
                    String scheduleIdRef = config.getAttribute(ATTR_SCHEDULE_ID_REF);

                    if(scheduleIdRef != null) {
                        instance.addListener((ScheduledEventListener) listener, scheduleIdRef);
                    } else {
                        long frequency = config.getLongAttribute(ATTR_FREQUENCY, 10);
                        instance.addListener((ScheduledEventListener) listener, frequency);
                    }
                }
            }
        } catch (SchedulingException e) {
            throw new ConfigurationException(e);
        }

        return instance;
    }

    private static List<Schedule> getSchedules(ScheduleProviderDocument.ScheduleProvider scheduleProviderConfig) throws ConfigurationException {
        List<Schedule> schedules = new ArrayList<Schedule>();
        List<org.jboss.soa.esb.listeners.config.xbeanmodel120.Schedule> scheduleConfigs = scheduleProviderConfig.getScheduleList();
        List<String> schedulIds = new ArrayList<String>();

        // We wouldn't have to do this if we were using JAXB!!!
        // TODO: Get XMLBeans to fudge the class names for the generated binds - existing names are potentialy dangerous because they're the same as the non XMLBeans types
        for(org.jboss.soa.esb.listeners.config.xbeanmodel120.Schedule scheduleConfig : scheduleConfigs) {
            String scheduleId = scheduleConfig.getScheduleid();
            Schedule schedule;

            if(schedulIds.contains(scheduleId)) {
                throw new ConfigurationException("Duplicate 'scheduleid' value of '" + scheduleId + "'.  Must be unique on a per <schedule-provider> basis.");
            }
            schedulIds.add(scheduleId);

            if(scheduleConfig instanceof SimpleScheduleDocument.SimpleSchedule) {
                SimpleSchedule simpleSchedule;
                FrequencyUnit.Enum frequencyUnit = ((SimpleScheduleDocument.SimpleSchedule)scheduleConfig).getFrequencyUnits();

                schedule = simpleSchedule = new SimpleSchedule(scheduleId);
                if(frequencyUnit == FrequencyUnit.SECONDS) {
                    simpleSchedule.setFrequency(((SimpleScheduleDocument.SimpleSchedule)scheduleConfig).getFrequency() * 1000);
                } else {
                simpleSchedule.setFrequency(((SimpleScheduleDocument.SimpleSchedule)scheduleConfig).getFrequency());
                }
                simpleSchedule.setExecCount(((SimpleScheduleDocument.SimpleSchedule)scheduleConfig).getExecCount());
            } else {
                CronSchedule cronSchedule;

                schedule = cronSchedule = new CronSchedule(scheduleId);
                cronSchedule.setCronExpression(((CronScheduleDocument.CronSchedule)scheduleConfig).getCronExpression());
            }
            schedule.setStartDate(scheduleConfig.getStartDate());
            schedule.setEndDate(scheduleConfig.getEndDate());
            schedules.add(schedule);
        }

        return schedules;
    }
}