/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.config.mappers;

import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.AbstractScheduledListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.FrequencyUnit;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.PropertyDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.Schedule;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.CronScheduleDocument.CronSchedule;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.ScheduleProviderDocument.ScheduleProvider;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.SimpleScheduleDocument.SimpleSchedule;
import org.jboss.soa.esb.schedule.SchedulerResource;
import org.w3c.dom.Element;

/**
 * Schedule Mapper.
 * <p/>
 * Maps the scheduling info from the Schedule based listener config, to the
 * ConfigTree config model.
 *
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ScheduleMapper {

    private static Logger logger = Logger.getLogger(ScheduleMapper.class);

    public static void map(Element listenerConfig, AbstractScheduledListener scheduledListener, final XMLBeansModel model) throws ConfigurationException {
        String scheduleIdRef = scheduledListener.getScheduleidref();
        Element domElement = (Element) scheduledListener.getDomNode();
        int scheduleFrequency = scheduledListener.getScheduleFrequency();
        int pollFrequency = scheduledListener.getPollFrequencySeconds();
        int pollLatency = getPollLatencySeconds(scheduledListener);

        if(scheduleIdRef != null) {
            listenerConfig.setAttribute(ListenerTagNames.SCHEDULE_ID_REF, scheduleIdRef);
            final Schedule schedule = model.getSchedule(scheduleIdRef) ;
            if (schedule.isSetStartDate()) {
                listenerConfig.setAttribute(ListenerTagNames.SCHEDULE_START_DATE, Long.toString(schedule.getStartDate().getTimeInMillis())) ;
            }
            if (schedule.isSetEndDate()) {
                listenerConfig.setAttribute(ListenerTagNames.SCHEDULE_END_DATE, Long.toString(schedule.getEndDate().getTimeInMillis())) ;
            }
            if (schedule instanceof SimpleSchedule) {
                final SimpleSchedule simpleSchedule = (SimpleSchedule)schedule ;
                final long simpleFrequency = simpleSchedule.getFrequency() ;
                final FrequencyUnit.Enum unit = simpleSchedule.getFrequencyUnits() ;
                final String frequencyVal ;
                if (unit.equals(FrequencyUnit.SECONDS)) {
                    frequencyVal = Long.toString(simpleFrequency * 1000) ;
                } else {
                    frequencyVal = Long.toString(simpleFrequency) ;
                }
                listenerConfig.setAttribute(ListenerTagNames.SCHEDULE_SIMPLE_FREQUENCY, frequencyVal) ;
                if (simpleSchedule.isSetExecCount()) {
                    listenerConfig.setAttribute(ListenerTagNames.SCHEDULE_SIMPLE_EXEC, Integer.toString(simpleSchedule.getExecCount())) ;
                }
            } else if (schedule instanceof CronSchedule) {
                final String cronExpression = ((CronSchedule)schedule).getCronExpression() ;
                if (cronExpression == null) {
                    throw new ConfigurationException("Missing cron expression from configuration") ;
                }
                listenerConfig.setAttribute(ListenerTagNames.SCHEDULE_CRON_EXPRESSION, cronExpression) ;
            } else {
                throw new ConfigurationException("Unknown schedule type specified in configuration: " + schedule.getClass().getName()) ;
            }
            
            if(domElement.hasAttribute("poll-frequency-seconds") || domElement.hasAttribute("schedule-frequency") ) {
                logger.warn("Schedule Listener '" + listenerConfig.getTagName() + "' defines both 'scheduleidref' and frequency attributes.  Using the 'scheduleidref'.");
            }
        } else if(scheduleFrequency > -1) {
            listenerConfig.setAttribute(ScheduleProviderFactory.ATTR_FREQUENCY, Integer.toString(scheduleFrequency));
        } else if(pollFrequency > -1) {
            listenerConfig.setAttribute(ScheduleProviderFactory.ATTR_FREQUENCY, Integer.toString(pollFrequency));
        } else if(pollLatency > -1) {
            listenerConfig.setAttribute(ScheduleProviderFactory.ATTR_FREQUENCY, Integer.toString(pollLatency));
        } else {
            listenerConfig.setAttribute(ScheduleProviderFactory.ATTR_FREQUENCY, "10");
        }

        final ScheduleProvider scheduleProvider = model.getScheduleProvider() ;
        if (scheduleProvider != null) {
            // There is only one schedule provider allowed in the configuration at present
            copyProperties(listenerConfig, model.getScheduleProvider().getPropertyList(), model.getScheduledListenerCount()) ;
        }
        
        if(domElement.hasAttribute("poll-frequency-seconds")) {
            logger.warn("Attrubute 'poll-frequency-seconds' is DEPRECATED.  Please change your configuration to use 'schedule-frequency'.");
        }
        if(pollLatency > -1) {
            logger.warn("Property 'pollLatencySeconds' is DEPRECATED.  Please change your configuration to use 'schedule-frequency'.");
        }
    }

    private static void copyProperties(final Element listenerConfig, final List<PropertyDocument.Property> properties, final int numScheduledListeners)
    {
        final Element schedulePropertyElement = YADOMUtil.addElement(listenerConfig, ListenerTagNames.SCHEDULE_PROPERTIES) ;
        schedulePropertyElement.setAttribute(SchedulerResource.THREAD_COUNT, Integer.toString(numScheduledListeners)) ;

        if ((properties != null) && (properties.size() > 0))
        {
            MapperUtil.mapProperties(properties, schedulePropertyElement) ;
        }
    }
    
    private static int getPollLatencySeconds(AbstractScheduledListener scheduledListener) {
        String value = XMLBeansModel.getProperty(scheduledListener.getPropertyList(), "pollLatencySeconds", "-1");

        try {
            return Integer.parseInt(value);
        } catch (NumberFormatException e) {
            return -1;
        }
    }
}
