/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.listeners.config;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.management.ObjectName;

import org.jboss.internal.soa.esb.listeners.war.Servlet;
import org.jboss.internal.soa.esb.publish.ContractProviderLifecycleResource;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.internal.soa.esb.publish.DefaultContractReferencePublisher;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.LifecycleUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;
import org.jboss.soa.esb.listeners.config.model.ModelParser;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleController;

/**
 * comment
 *
 * @author <a href="kurt.stam@jboss.com">Kurt Stam</a>
 * @author <a href="bill@jboss.com">Bill Burke</a>
 * @version $Revision: 1.1 $
 */
public class Configuration
{
   public static String getStringFromStream(InputStream is) throws Exception
   {
      byte[] bytes = StreamUtils.readStream(is);
      return new String(bytes, "UTF-8");
   }

   public static ManagedLifecycleController create(URL jbossEsbXml)
   {
      String configXml = null;
      try
      {
         configXml = getStringFromStream(jbossEsbXml.openStream());
         return create(configXml);
      }
      catch (Exception e)
      {
         throw new RuntimeException("Failed to load jbossesb.xml file: " + jbossEsbXml, e);
      }
   }

   public static String getMep(ConfigTree listenerConfig) {
       String mep = listenerConfig.getAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG);
       if(mep != null) {
           return mep;
       } else {
           return ListenerTagNames.MEP_REQUEST_RESPONSE;
       }
   }

   public static ManagedLifecycleController create(String configXml) {
	   return create (configXml, null);
   }
   
   public static ManagedLifecycleController create(String configXml, ObjectName serviceName)
   {
       return create(configXml, serviceName, null) ;
   }
   
   public static ManagedLifecycleController create(final String configXml, final ObjectName serviceName, final List<ContractReferencePublisher> publishers)
   {
	   return create(configXml, serviceName, publishers, null);
   }
   
   public static ManagedLifecycleController create(final String configXml, final ObjectName serviceName, List<ContractReferencePublisher> publishers, final List<Servlet> servlets)
   {
	  if (publishers == null)
	  {
		  publishers = new ArrayList<ContractReferencePublisher>();
	  }
      try
      {
         final Reader config = new StringReader(configXml);
         if (ModelParser.getParser().validate(config))
         {
            ByteArrayOutputStream listenerXml = new ByteArrayOutputStream();
            ByteArrayOutputStream gatewayXml = new ByteArrayOutputStream();
            Generator generator = new Generator(new ByteArrayInputStream(configXml.getBytes()), listenerXml, gatewayXml);
            generator.generate();
            byte[] listenerBytes = listenerXml.toByteArray();
            //System.out.println("*************** listenerBytes: " + new String(listenerBytes));
            ByteArrayInputStream listenerIs = new ByteArrayInputStream(listenerBytes);
            ConfigTree listenerConfig = ConfigTree.fromInputStream(listenerIs);
            if (serviceName != null) {
            	listenerConfig.setAttribute(ListenerTagNames.DEPLOYMENT_NAME_TAG, 
            		serviceName.getKeyProperty(ListenerTagNames.DEPLOYMENT_NAME_TAG));
            }
            List<ManagedLifecycle> instances = LifecycleUtil.getListeners(listenerConfig);

            byte[] gatewayBytes = gatewayXml.toByteArray();
            //System.out.println("*************** gatewayBytes: " + new String(gatewayBytes));
            ByteArrayInputStream gatewayIs = new ByteArrayInputStream(gatewayBytes);
            ConfigTree gatewayConfig = ConfigTree.fromInputStream(gatewayIs);
            instances.addAll(LifecycleUtil.getGateways(gatewayConfig));
            
            ModelAdapter model = generator.getModel();
            
            // Handle the ServiceContracts
            if (servlets != null && !servlets.isEmpty())
            {
	        	List<ServiceContract> modelServiceContracts = model.getServiceContracts();
	        	if (modelServiceContracts != null && modelServiceContracts.size() > 0) {
		            for (Servlet servlet : servlets) {
		            	Service service = servlet.getService();
		            	String endpointAddress = servlet.getEndpointAddress();
		            	if (service != null && endpointAddress != null) {
		            		for (ServiceContract serviceContract : modelServiceContracts) {
		            			if (service.equals(serviceContract.getService())) {
		            				publishers.add(
		            					new DefaultContractReferencePublisher(
		            						service, serviceContract.getDescription(), endpointAddress ) );
		            				ContractProviderLifecycleResource.putContractProvider(
		            						service.getCategory(), service.getName(), serviceContract.getContractProvider() );
		            				break; // first ServiceContract matching a Servlet wins!
		            			}
		            		}
		            	}
		            }
	        	}
            }

             ManagedLifecycleController controller = new ManagedLifecycleController(instances);

             // In parallel, create a map of the contract publication info...
             boolean success = false ;
             try
             {
                 ServicePublisher.addServicePublishers(controller, model.getServicePublishers());
                 ServicePublisher.addContractReferencePublishers(controller, publishers) ;
                 success = true ;
             }
             finally
             {
                 if (!success)
                 {
                     ServicePublisher.removeServicePublishers(controller) ;
                 }
             }
            return controller;
         }
         else
         {
            throw new IllegalStateException("ESB file had validation errors.");

         }
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }
}
