/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.http;

import java.security.Principal;
import java.security.cert.X509Certificate;
import java.util.HashSet;
import java.util.Set;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.security.PublicCryptoUtil;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.jboss.util.Base64;

/**
 * Http Servlet security utilities.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class HttpServletSecUtil {

    private static final Logger logger = Logger.getLogger(HttpServletSecUtil.class);

    public static boolean addAuthDetailsToMessage(HttpServletRequest request, Message message) throws ServletException {

            // Need to get the login details from the request
    		// and pass it along in the ESB Message sent to the Service...
            AuthenticationRequest authRequest = buildAuthenticationRequest(request);

            if (authRequest != null) {
            	// It's a logged in user. Set the AuthenticationRequest on the inbound message...
	            try {
	                return PublicCryptoUtil.INSTANCE.addAuthRequestToMessage(authRequest, message);
	            } catch (MessageDeliverException e) {
	                throw new ServletException("Failed to attach AuthenticationRequest to ESB Message.", e);
	            }
            }

        return false;
    }

    public static AuthenticationRequest buildAuthenticationRequest(HttpServletRequest req) throws ServletException {
    	String authType = req.getAuthType();

    	// Handle BASIC auth...
    	if (HttpServletRequest.BASIC_AUTH.equals(authType)) {
    		if (req.getUserPrincipal() != null) {
    			String authHeader = req.getHeader("Authorization");
    			if(authHeader == null) {
    				authHeader = req.getHeader("authorization");
    			}
    			if(authHeader != null) {
    				String bas64Data = authHeader.substring(authHeader.indexOf(' ')).trim();
    				String decodedData = new String(Base64.decode(bas64Data));
    				String[] authTokens = decodedData.split(":");
    				Set<Object> credentials = new HashSet<Object>();

    				// The auth tokens are "username:password", so the second token is the
    				// password, which is the credential in this case...
    				credentials.add(authTokens[1].toCharArray());

    				if(logger.isDebugEnabled()) {
    					logger.debug("User '" + authTokens[0] + "' authenticated by container using '" + HttpServletRequest.BASIC_AUTH + "' auth.  Forwarding to ESB Service.");
    				}
    				return new AuthenticationRequestImpl.Builder(req.getUserPrincipal(), credentials).build();
    			}
    		}
    	}
    	// Handle CLIENT_CERT auth...
    	else if (HttpServletRequest.CLIENT_CERT_AUTH.equals(authType)) {
    		Principal principal = null;
    		Set<Object> credentials = null;
    		
    		Object attr = req.getAttribute("javax.servlet.request.X509Certificate");
    		if (attr != null) {
    			X509Certificate[] chain;
    			if (attr.getClass().isArray()) {
    				chain = (X509Certificate[])attr;
    			} else {
    				chain = new X509Certificate[]{(X509Certificate)attr};
    			}
    			for (X509Certificate cert : chain) {
    				if (cert != null) {
    					principal = cert.getSubjectX500Principal();
    					credentials = new HashSet<Object>();
    					credentials.add(cert);
    					break;
    				}
    			}
    		}
    		
    		if (principal == null) {
    			if(logger.isDebugEnabled()) {
    				logger.debug(HttpServletRequest.CLIENT_CERT_AUTH + " specified but principal not available.");
    			}
    			return null;
    		} else if (credentials == null) {
    			if(logger.isDebugEnabled()) {
    				logger.debug(HttpServletRequest.CLIENT_CERT_AUTH + " specified but credentials not available.");
    			}
    			return null;
    		}

    		if(logger.isDebugEnabled()) {
    			logger.debug("User '" + principal.getName() + "' authenticated by container using '" + HttpServletRequest.CLIENT_CERT_AUTH + "' auth.  Forwarding to ESB Service.");
    		}
    		return new AuthenticationRequestImpl.Builder(principal, credentials).build();
    		
    	} else if (authType != null && logger.isDebugEnabled()) {
    		logger.debug("Authentication  method '" + authType + "' not supported for passing Authorization token to ESB Security.");
    	}
    	// TODO: Add support for Digest auth?

    	return null;
    }
}
