/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

package org.jboss.soa.esb.addressing.util;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.Hashtable;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.jboss.internal.soa.esb.addressing.helpers.EPRHelper;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.soa.esb.addressing.EPR;

/**
 * Allows EPRs to be saved to and loaded from files. Mainly for testing
 * purposes. This version assumes XML as the output format.
 * 
 * @author marklittle
 * 
 */

public class EPRManager
{

	/**
	 * All EPRs are saves in files within the current working directory.
	 * 
	 * @return the manager for the cwd.
	 */

	public static final EPRManager getInstance()
	{
		return getInstance("");
	}

	/**
	 * All EPRs are saves in files within a defined directory. Get the right
	 * manager for that directory.
	 * 
	 * @param domain
	 *            the name of the directory. If <code>null</code> then the
	 *            null String is assumed.
	 * @return the manager for the directory. If it does not exist, then one
	 *         will be created.
	 */

	public static final EPRManager getInstance(String domain)
	{
		if (domain == null)
			domain = "";

		synchronized (_instances)
		{
			EPRManager theInstance = _instances.get(domain);

			if (theInstance == null)
			{
				theInstance = new EPRManager(domain);

				_instances.put(domain, theInstance);
			}

			return theInstance;
		}
	}

	public final String getDomain()
	{
		return _directory;
	}

	public boolean equals(Object manager)
	{
		if (manager instanceof EPRManager)
		{
			EPRManager comp = (EPRManager) manager;

			if (_directory.equals(comp.getDomain()))
				return true;
		}

		return false;
	}
	
	/**
	 * Return a hash code for this element.
	 * @return the element hash code.
	 */
	@Override
	public int hashCode()
	{
		return _directory.hashCode() ;
	}

	/**
	 * Save the EPR into the specified file.
	 * 
	 * @param name
	 *            the name of the file to use (the logical service name).
	 * @param address
	 *            the EPR to save.
	 * 
	 * @throws IOException
	 *             thrown if there is an error.
	 */

	public final void saveEPR(String name, EPR address) throws IOException
	{
		if ((name == null) || (address == null))
			throw new IllegalArgumentException();

		try
		{
			String documentAsString = EPRHelper.toXMLString(address);

			FileOutputStream output = new FileOutputStream(_directory
					+ File.separator + name);

                        try
                        {
                            output.write(documentAsString.getBytes());
                            output.flush();
                            output.getFD().sync(); // make sure it's on disk!
                        }
                        finally
                        {
                            output.close() ;
                        }
		}
		catch (Exception ex)
		{
			throw new IOException(ex.toString());
		}
	}

	/**
	 * Remove the EPR-to-file association.
	 * 
	 * @param name
	 *            the logical name for the service.
	 * @throws IOException
	 *             thrown if there are any errors.
	 */

	public final void removeEPR(String name) throws IOException
	{
		if (name == null)
			throw new IllegalArgumentException();

		File theFile = new File(_directory + File.separator + name);

		if (theFile.exists())
			theFile.delete();
		else
			throw new FileNotFoundException();
	}

	/**
	 * Get the EPR specified by the logical name.
	 * 
	 * @param name
	 *            the service name.
	 * @return the EPR, or <code>null</code> if none exists.
	 * @throws IOException
	 *             thrown if there is an error.
	 */

	public final EPR loadEPR(String name) throws IOException
	{
		if (name == null)
			throw new IllegalArgumentException();

		File theFile = new File(_directory + File.separator + name);

		try
		{
			if (theFile.exists())
			{
			    final FileReader reader = new FileReader(theFile) ;
			    try
			    {
			        final XMLStreamReader in = XMLHelper.getXMLStreamReader(reader) ;
			        StreamHelper.skipToNextStartElement(in) ;

				return EPRHelper.fromXML(in);
			    }
			    finally
			    {
			        reader.close() ;
			    }
			}
			else
				throw new FileNotFoundException(theFile.toString());
		}
		catch (XMLStreamException xmlse)
		{
		    final IOException ioe = new IOException("Failed to parse EPR file: " + theFile.getAbsolutePath()) ;
		    ioe.initCause(xmlse) ;
		    throw ioe ;
		}
	}

	protected EPRManager(String domain)
	{
		_directory = domain;
	}

	private String _directory;

	private static Hashtable<String, EPRManager> _instances = new Hashtable<String, EPRManager>();

}