/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.converters;


import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.util.ClassUtil;


/**
 * Object to processor.
 * <p/>
 * Uses the <a href="http://xstream.codehaus.org/">XStream</a> processor to generate an XML message String from the supplied object.
 * <p/>
 * Sample Action Configuration:
 * <pre>
 * &lt;Action name="doCustomer" processor="ObjectInvoke"&gt;
 *     &lt;property name="class-processor" value="CustomerProcessor" /&gt; &lt;!-- Required. Class processor used to process the message . --&gt;
 *     &lt;property name="class-method" value="fooMethodName" /&gt; &lt;!-- Optional. Default to name of action. --&gt;
 * 
 * &lt;/Action&gt;
 * </pre>
 * <p/>
 * 
 * This can be used with ObjectToXStream and XStreamToObject
 * 
 * @author danielmarchant
 * @since Version 4.0
 */
public class ObjectInvoke extends AbstractActionPipelineProcessor {

	private static Logger logger = Logger.getLogger(ObjectInvoke.class);
	
	private String classProcessor;
    
    // class related variables
    private Class processorClass;
    private Method method;
    private String classMethod;
    
    // action related variables
    private String name;
    private MessagePayloadProxy payloadProxy;

    /**
     * Public constructor.
     * @param configTree Action Properties.
     * @throws ConfigurationException Action not properly configured.
     */
    public ObjectInvoke(ConfigTree configTree) {
        payloadProxy = new MessagePayloadProxy(configTree,
                                               new String[] {BytesBody.BYTES_LOCATION, ActionUtils.POST_ACTION_DATA},
                                               new String[] {ActionUtils.POST_ACTION_DATA});

    	name = configTree.getName();
        classProcessor = configTree.getAttribute("class-processor");
        classMethod = configTree.getAttribute("class-method");
        if((classMethod == null) || (classMethod.length() == 0) ) {
        	classMethod = name;
        }
        
        if(classProcessor == null) {
        	return;
        }
        
        try {
			processorClass = ClassUtil.forName(classProcessor, getClass());
			
			this.setUpMethod(processorClass);
			
		} catch (ClassNotFoundException e) {
			logger.error("Not able to find class: " + classProcessor, e);
		}
    }
	
	
	/**
	 * Processes the message by using the giving class-processor.
	 *  
	 */
	public Message process(Message message) throws ActionProcessingException {
        Object object = null;

        try {
            object = payloadProxy.getPayload(message);
        } catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }

        Object processor = getObjectToInvoke();
		Method m = this.setUpMethod(processor);
		
		Object result;
		try {
			result = m.invoke(processor, new Object[] { object });
		} catch (IllegalArgumentException e) {
			throw new ActionProcessingException("Invalid arguments for class-processor [ " + this.classProcessor + "] for Action Processor: " + name, e );
			
		} catch (IllegalAccessException e) {
			throw new ActionProcessingException("No access for class-processor [ " + this.classProcessor + "] for Action Processor: " + name, e );
			
		} catch (InvocationTargetException e) {
			throw new ActionProcessingException("Invocation problem with class-processor [ " + this.classProcessor + "] for Action Processor: " + name, e );
			
		}

        try {
            payloadProxy.setPayload(message, result);
        } catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }
        
        return message;
	}

	
	/**
	 * Gets the object that will process the message.
	 * Classes can extend this to provide the object in the way they want.
	 * 
	 * 
	 * @return Object
	 * @throws ActionProcessingException
	 */
	protected Object getObjectToInvoke()  throws ActionProcessingException {
		Object processor = null;
		try {
			processor = processorClass.newInstance();
		} catch (InstantiationException e) {
			throw new ActionProcessingException("Could not invoke class-processor [ " + this.classProcessor + "] for Action Processor: " + name, e );
		} catch (IllegalAccessException e) {
			throw new ActionProcessingException("Could not access class-processor [ " + this.classProcessor + "] for Action Processor: " + name, e );
		}
		return processor;
	}
	
	
	/**
	 * Gets the method to invoke on object.
	 * 
	 * @param obj
	 * @return Method
	 */
	protected Method setUpMethod(Object obj) {
		if(method != null) 
		{
			return method;
		}
		
		if(obj instanceof Class) {
			processorClass = (Class)obj;
		} else {
			processorClass = obj.getClass();
		}

		Method[] methods = processorClass.getMethods();
		for(int i=0; i<methods.length; i++) {
			Method m = methods[i];
			if(m.getName().equals(classMethod)) {
				this.method = m;
				break;
			}
		}
		
		return method;
	}


}
