/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006
 */
package org.jboss.soa.esb.actions;

import java.util.ArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Simple action that passes the message to a logging mechanism.
 * This action is primarily used for testing the quickstarts.
 * 
 * @author Kevin Conner
 */
public class TestMessageStoreSink
{
    /**
     * The list of messages.
     */
    private final ArrayList<String> list = new ArrayList<String>() ;
    /**
     * The lock for the sink.
     */
    private final Lock lock = new ReentrantLock() ;
    /**
     * The condition for notifying messages.
     */
    private final Condition waitCondition = lock.newCondition() ;
    
    /**
     * Wait for the specified number of messages to appear in the sink.
     * @param numberOfMessages The number of messages.
     * @param maxDelay The maximum delay for waiting.
     */
    public String[] waitForMessages(final int numberOfMessages, final long maxDelay)
    {
        final long startTime = System.currentTimeMillis() ;
        final long endTime = startTime + maxDelay ;
        lock.lock() ;
        try
        {
            while (list.size() != numberOfMessages)
            {
                final long now = System.currentTimeMillis() ;
                final long waitPeriod = endTime - now ;
                if (waitPeriod > 0)
                {
                    try
                    {
                        waitCondition.await(waitPeriod, TimeUnit.MILLISECONDS) ;
                    }
                    catch (final InterruptedException ie) {} // ignore
                }
                else
                {
                    break ;
                }
            }
            final String[] results = list.toArray(new String[0]) ;
            list.clear() ;
            final int numResults = results.length ;
            System.out.println("TestMessageStoreSink: Returning " + numResults + " message" + (numResults == 1 ? "" : "s")) ;
            for(int count = 0 ; count < numResults ; count++)
            {
                System.out.println("TestMessageStoreSink: Returning message[" + count + "]: " + results[count]) ;
            }
            return results ;
        }
        finally
        {
            lock.unlock() ;
        }
    }

    /**
     * Log the message in the sink.
     * @param message The current message.
     */
    public void logMessage(final String message)
    {
        lock.lock() ;
        try
        {
            list.add(message) ;
            System.out.println("TestMessageStoreSink: Adding message: " + message) ;
            waitCondition.signal() ;
        }
        finally
        {
            lock.unlock() ;
        }
    }

    /**
     * Clear the messages in the sink.
     */
    public void clearMessages()
    {
        lock.lock() ;
        try
        {
            list.clear() ;
            System.out.println("TestMessageStoreSink: Clearing messages") ;
        }
        finally
        {
            lock.unlock() ;
        }
    }
}
