/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.net.URI;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.util.ClassUtil;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;

/**
 * SchemaResolver is a {@link LSResourceResolver} implemention that locates
 * resources by first trying to locate the resource on the local filesystem, 
 * and failing to find it there it will try the classpath. 
 * </p>
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class SchemaResolver implements LSResourceResolver
{
    private Logger log = Logger.getLogger(SchemaResolver.class);

    /**
     * The encoding to use when reading resources.
     */
    final private String encoding;

    /**
     * Base URI that resources will be resolved realtive to.
     */
    private final URI parentSchemaUri;

    /**
     * Constructs an instance with the default encoding (UTF-8) and
     * using the passed in baseUri.
     * 
     * @param baseUri Base URI that resources will be resolved realtive to.
     */
    public SchemaResolver(final URI baseUri)
    {
        this("UTF-8", baseUri);
    }

    /**
     * Constructs an instance with the specified encoding.
     * 
     * @param encoding The encoding to use when reading resources. Must not be null;
     * @param baseUri Base URI that resources will be resolved realtive to.
     */
    public SchemaResolver(final String encoding, final URI baseUri)
    {
        AssertArgument.isNotNullAndNotEmpty(encoding, "encoding");
        AssertArgument.isNotNull(baseUri, "baseUri");
        this.encoding = encoding;
        this.parentSchemaUri = baseUri;
    }
    
    /**
     * Tries to resolve the resource.
     * 
     * @return LSInput  A new {@link LSInputImpl} object instance.
     */
    public LSInput resolveResource(final String type, final String namespaceURI, final String publicId, String systemId, final String baseURI)
    {
        if (log.isDebugEnabled())
        {
            log.debug(String.format("parentSchemaUri='%s', systemId='%s', baseURI='%s'", this.parentSchemaUri, systemId, baseURI));
        }
        
        LSInputImpl inputImpl = new LSInputImpl();

        URI resolvedUri;
        if (baseURI != null)
        {
            // Resolve relative to the baseUri passed in. 
            URI uri = URI.create(baseURI);
            log.debug("Try to resolve " + uri + "/" + systemId);
            resolvedUri = uri.resolve(systemId);
        }
        else
        {
            // Resolve the systemId (the schema path) relative to the parent schema.
            resolvedUri = parentSchemaUri.resolve(systemId);
        }
        log.debug("Resolved Schema '" + resolvedUri + "'");
        
        inputImpl.setSystemId(resolvedUri.toString());
        inputImpl.setCharacterStream(getReader(resolvedUri));
        return inputImpl;
    }
    
    private Reader getReader(final URI resolvedUri)
    {
        String scheme = resolvedUri.getScheme();
        if (resolvedUri.isAbsolute())
        {
            if (scheme.equals("file"))
            {
                try
                {
                    // Try to locate the resource from the local filesystem.
                    InputStream in = getFileInputStream(resolvedUri);
                    return new StringReader(StreamUtils.readStreamString(in, encoding));
                } 
                catch (final FileNotFoundException e)
                {
                    log.error(e.getMessage(), e);
                } 
                catch (UnsupportedEncodingException e)
                {
                    log.error(e.getMessage(), e);
                }
            }
            else if (scheme.equals("classpath"))
            {
                InputStream in = ClassUtil.getResourceAsStream(resolvedUri.getPath(), getClass());
                if (in != null)
                {
                    try
                    {
                        return new StringReader(StreamUtils.readStreamString(in, encoding));
                    } 
                    catch (final UnsupportedEncodingException e)
                    {
                        log.error(e.getMessage(), e);
                    }
                }
                else
                {
                    log.info("Could not read resource : " + resolvedUri.getPath());
                }
            }
        }
        return null;
    }

    private class LSInputImpl implements LSInput
    {
        private String publicId;
        private String systemId;
        private String baseUri;
        private Reader reader;
        
        public LSInputImpl()
        {
        }
        
        public Reader getCharacterStream()
        {
            return reader;
        }

        /**
         * @return null
         */
        public InputStream getByteStream()
        {
            return null;
        }

        /**
         * @return null
         */
        public String getStringData()
        {
            return null;
        }

        public String getSystemId()
        {
            return systemId;
        }

        public String getBaseURI()
        {
            return baseUri;
        }

        public String getPublicId()
        {
            return publicId;
        }

        public String getEncoding()
        {
            return encoding;
        }

        /**
         * @return false
         */
        public boolean getCertifiedText()
        {
            return false;
        }

        public void setBaseURI(String baseURI)
        {
            this.baseUri = baseURI;
        }

        public void setByteStream(InputStream byteStream)
        {
        }

        /**
         * @throws UnsupportedOperationException
         */
        public void setCertifiedText(boolean certifiedText)
        {
            throw new UnsupportedOperationException("setCertifiedText is not supported");
        }

        /**
         * @throws UnsupportedOperationException
         */
        public void setCharacterStream(Reader characterStream)
        {
            this.reader = characterStream;
        }

        /**
         * @throws UnsupportedOperationException
         */
        public void setEncoding(String encoding)
        {
            throw new UnsupportedOperationException("setEncoding is not supported");
        }

        public void setPublicId(String publicId)
        {
            this.publicId = publicId;
        }

        /**
         * @throws UnsupportedOperationException
         */
        public void setStringData(String stringData)
        {
            throw new UnsupportedOperationException("setStringData is not supported");
        }

        public void setSystemId(String systemId)
        {
            this.systemId = systemId;
        }
    }

    private InputStream getFileInputStream(final URI uri) throws FileNotFoundException
    {
        File file = new File(uri);
        if (file.exists() && !file.isDirectory())
        {
            return new FileInputStream(file);
        }

        throw new FileNotFoundException("Could not locate '" + uri + "' on local filesystem");
    }
}
