/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.notification;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;


/**
 * MacroExpander class.
 * <p/>
 * This class takes a DOM and iterates over it, replacing all "macro" tokens in 
 * any attribute values using the values from the supplied Map.
 * 
 * @author <a href="mailto:schifest@gmail.com">Esteban Shifman</a>
 */
public class MacroExpander {
	private static final String CTX_PREFIX = "@@ctx@@";

	public static final String CTX_BATCH_UID = CTX_PREFIX + "batchUid";

	public static final String CTX_BATCH_NUM = CTX_PREFIX + "batchNum";

	public static final String CTX_INPUT_FILE = CTX_PREFIX + "inputFile";

	public static final String CTX_POST_FILE = CTX_PREFIX + "postFile";

	private static final String[] s_saMacros = { 
		CTX_BATCH_UID, CTX_BATCH_NUM, CTX_INPUT_FILE, CTX_POST_FILE };

	/**
	 * Perform a macro replacement on the DOM tree attribute values
	 * of the supplied ConfigTree using the supplied macro token
	 * replacement map.
	 * @param domTree The DOM Tree to be operated on.
	 * @param replacementTokenMapMap The token replacement map.
	 */
	public static void replaceMacros(ConfigTree domTree, Map replacementTokenMap) {

		List<KeyValuePair> changed = new ArrayList<KeyValuePair>();
		for (String currAtt : domTree.getAttributeNames()) 
		{
			String sVal = domTree.getAttribute(currAtt);
			boolean bSubst = false;
			int iPos = 0;

			while (0 <= (iPos = sVal.indexOf(CTX_PREFIX, 0))) {
				String sRest = sVal.substring(iPos);
				for (int ii = 0; ii < s_saMacros.length; ii++) {
					String sMac = s_saMacros[ii];
					if (!sRest.startsWith(sMac)) {
						continue;
					}
					Object oRpl = (null != replacementTokenMap) ? replacementTokenMap.get(sMac) : null;
					if (null == oRpl) {
						continue;
					}
					bSubst = true;
					sVal = sVal.substring(0, iPos) + oRpl.toString() + sVal.substring(iPos + sMac.length());
					break;
				}
			}
			if (bSubst)
					changed.add(new KeyValuePair(currAtt,sVal));
		}
		for (KeyValuePair kvp : changed)
			domTree.setAttribute(kvp.getKey(),kvp.getValue());
		changed	= null;
		
		// Iterate over the child elements and recursively call this method...
		for (ConfigTree child : domTree.getAllChildren())
				replaceMacros(child, replacementTokenMap);
	}
}
