/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.message.format;

import org.jboss.internal.soa.esb.message.format.serialized.JavaMessageSerializer;
import org.jboss.internal.soa.esb.message.format.xml.XMLMessageSerializer;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.message.Message;

import java.io.*;

/**
 * Message Serialization interface.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class MessageSerializer {

    /**
     * XML message byte buffer preamble.
     */
    public static final byte PREAMBLE_XML = 0;
    /**
     * Java message byte buffer preamble.
     */
    public static final byte PREAMBLE_JAVA = 1;

    private static XMLMessageSerializer xmlSerializer = new XMLMessageSerializer();
    private static JavaMessageSerializer javaSerializer = new JavaMessageSerializer();

    /**
     * Serialize the supplied message to the supplied {@link OutputStream}.
     *
     * @param message   The message to be serialized.
     * @param outStream The target {@link OutputStream}.
     */
    public abstract void serialize(Message message, OutputStream outStream) throws IOException;

    /**
     * Deserialize the message supplied in the {@link InputStream}.
     *
     * @param inStream The message source {@link InputStream}.
     */
    public abstract Message deserialize(InputStream inStream) throws IOException;

    public static byte[] serialize(Message message) throws IOException {
        ByteArrayOutputStream bytesOutStream = new ByteArrayOutputStream();
        MessageSerializer.serializeMessage(message, bytesOutStream);

        return bytesOutStream.toByteArray();
    }

    public static Message deserialize(byte[] messageBytes) throws IOException {
        ByteArrayInputStream bytesInStream = new ByteArrayInputStream(messageBytes);
        return MessageSerializer.deserializeMessage(bytesInStream);
    }

    public static void serializeMessage(Message message, OutputStream outStream) throws IOException {
        AssertArgument.isNotNull(message, "message");
        AssertArgument.isNotNull(outStream, "outStream");

        MessageSerializer serializer;

        // Select the serializer...
        if(XMLMessageSerializer.isXMLMessage(message)) {
            serializer = xmlSerializer;
        } else if(JavaMessageSerializer.isJavaMessage(message)) {
            serializer = javaSerializer;
        } else {
            throw new IOException("Unsupported Message implementation type '" + message.getClass().getName() + "'.");
        }

        // Run the serialize...
        serializer.serialize(message, outStream);
    }

    public static Message deserializeMessage(InputStream inStream) throws IOException {
        AssertArgument.isNotNull(inStream, "inStream");

        if(!inStream.markSupported()) {
            throw new IOException("Cannot perform Message deserialization on supplied InputStream.  The InputStream must support mark and reset!");
        }
        inStream.mark(1);

        int preamble = inStream.read();
        MessageSerializer serializer;

        inStream.reset();
        if(preamble == PREAMBLE_XML) {
            serializer = xmlSerializer;
        } else if(preamble == PREAMBLE_JAVA) {
            serializer = javaSerializer;
        } else {
            throw new IOException("Unsupported serialized Message format. Unrecognized message preamble.");
        }

        return serializer.deserialize(inStream);
    }
}
