/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.bpel.samples.quickstart.purchase;

import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.Resource;
import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.JMSException;
import javax.jms.MapMessage;
import javax.jms.MessageProducer;
import javax.jms.Queue;
import javax.jms.Session;
import javax.jws.WebService;

/**
 * Shipping endpoint bean.
 * 
 * @author Jeff DeLong
 * @author Alejandro Guizar
 */
@WebService(endpointInterface = "org.jboss.soa.bpel.samples.quickstart.purchase.ShippingPT", targetNamespace = "http://manufacturing.org/wsdl/purchase", serviceName = "shippingSVC", portName = "shippingSP")
public class ShippingPT_Impl implements ShippingPT {

	@Resource
	private Queue shippingQueue;
	@Resource
	private ConnectionFactory connectionFactory;
	@Resource
	private Float shippingPrice;

	private final Logger logger =
			Logger.getLogger(ShippingPT_Impl.class.getName());

	/**
	 * Returns shipping price synchronously and date asynchronously.
	 */
	public ShippingInfo requestShipping(CustomerInfo customerInfo)
			throws OrderFaultType {
		// check the address before accepting the shipping
		String address = customerInfo.getAddress();
		if (address.indexOf("Elm St") != -1) {
			// we do not ship to Elm St, it is a scary place
			ProblemInfo problemInfo = new ProblemInfo();
			problemInfo.setDetail("Shipping to address '" + address
					+ "' is not available");
			throw new OrderFaultType("shipping unavailable", problemInfo);
		}

		try {
			sendShippingMessage(customerInfo.getCustomerId());
		}
		catch (JMSException e) {
			logger.log(Level.SEVERE, "failed to send shipping message", e);
		}

		ShippingInfo shippingInfo = new ShippingInfo();
		shippingInfo.setShippingPrice(shippingPrice);
		return shippingInfo;
	}

	protected void sendShippingMessage(String customerId) throws JMSException {
		Connection connection = connectionFactory.createConnection();
		try {
			// create a session
			Session session =
					connection.createSession(false, Session.CLIENT_ACKNOWLEDGE);
			try {
				// create the message
				MapMessage message = session.createMapMessage();
				message.setString("customerId", customerId);

				// send it!
				MessageProducer producer = session.createProducer(shippingQueue);
				producer.send(message);

				logger.log(Level.FINE, "sent shipping message for customer {0}",
						customerId);
			}
			finally {
				session.close();
			}
		}
		finally {
			connection.close();
		}
	}
}
