/*
 * RHQ Management Platform
 * Copyright (C) 2005-2010 Red Hat, Inc.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package org.rhq.enterprise.server.core.jaas;

import java.security.acl.Group;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.FailedLoginException;
import javax.security.auth.login.LoginException;
import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jboss.crypto.CryptoUtil;
import org.jboss.security.SimpleGroup;
import org.jboss.security.SimplePrincipal;
import org.jboss.security.auth.spi.UsernamePasswordLoginModule;

import org.rhq.enterprise.server.RHQConstants;

/**
 * A JDBC login module that passes only if there is no principal in the database.
 */

public class JDBCPrincipalCheckLoginModule extends UsernamePasswordLoginModule {
    private Log log = LogFactory.getLog(JDBCPrincipalCheckLoginModule.class);
    private String dsJndiName;
    private String principalsQuery = "SELECT password FROM RHQ_PRINCIPAL WHERE principal=?";

    @Override
    public void initialize(Subject subj, CallbackHandler handler, Map shared_state, Map opts) {
        super.initialize(subj, handler, shared_state, opts);

        dsJndiName = (String) opts.get("dsJndiName");
        if (dsJndiName == null) {
            dsJndiName = RHQConstants.DATASOURCE_JNDI_NAME;
        }

        Object tmpQuery = opts.get("principalsQuery");

        if (tmpQuery != null) {
            principalsQuery = tmpQuery.toString();
        }

        log.debug("dsJndiName=" + dsJndiName);
        log.debug("prinipalsQuery=" + principalsQuery);
    }

    /**
     * @see org.jboss.security.auth.spi.UsernamePasswordLoginModule#getUsersPassword()
     */
    @Override
    protected String getUsersPassword() throws LoginException {
        String username = getUsername();
        if ("admin".equals(username)) {
            throw new FailedLoginException("Cannot log in as overlord");
        }
        String password = getUsernameAndPassword()[1]; // what did the user enter?
        Connection conn = null;
        PreparedStatement ps = null;
        ResultSet rs = null;

        try {
            InitialContext ctx = new InitialContext();
            DataSource ds = (DataSource) ctx.lookup(dsJndiName);
            conn = ds.getConnection();

            ps = conn.prepareStatement(principalsQuery);
            ps.setString(1, username);
            rs = ps.executeQuery();
            if (rs.next() == true) {
                throw new FailedLoginException("username found in principals - do not continue");
            }

            password = CryptoUtil.createPasswordHash("MD5", "base64", null, null, password); // return back the string entered by the user as a hash
        } catch (NamingException ex) {
            throw new LoginException(ex.toString(true));
        } catch (SQLException ex) {
            throw new LoginException(ex.toString());
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (Exception e) {
                }
            }

            if (ps != null) {
                try {
                    ps.close();
                } catch (Exception e) {
                }
            }

            if (conn != null) {
                try {
                    conn.close();
                } catch (Exception ex) {
                }
            }
        }

        return password;
    }

    /**
     * @see org.jboss.security.auth.spi.AbstractServerLoginModule#getRoleSets()
     */
    @Override
    protected Group[] getRoleSets() throws LoginException {
        SimpleGroup roles = new SimpleGroup("Roles");

        //roles.addMember( new SimplePrincipal( "some user" ) );
        Group[] roleSets = { roles };
        return roleSets;
    }
}