/**
 * JDBM LICENSE v1.00
 *
 * Redistribution and use of this software and associated documentation
 * ("Software"), with or without modification, are permitted provided
 * that the following conditions are met:
 *
 * 1. Redistributions of source code must retain copyright
 *    statements and notices.  Redistributions must also contain a
 *    copy of this document.
 *
 * 2. Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *
 * 3. The name "JDBM" must not be used to endorse or promote
 *    products derived from this Software without prior written
 *    permission of Cees de Groot.  For written permission,
 *    please contact cg@cdegroot.com.
 *
 * 4. Products derived from this Software may not be called "JDBM"
 *    nor may "JDBM" appear in their names without prior written
 *    permission of Cees de Groot.
 *
 * 5. Due credit should be given to the JDBM Project
 *    (http://jdbm.sourceforge.net/).
 *
 * THIS SOFTWARE IS PROVIDED BY THE JDBM PROJECT AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CEES DE GROOT OR ANY CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright 2001 (C) Alex Boisvert. All Rights Reserved.
 * Contributions are Copyright (C) 2001 by their associated contributors.
 *
 */

package jdbm.helper;

import java.util.Comparator;
import java.io.Serializable;

import org.apache.directory.server.i18n.I18n;

/**
 * Comparator for byte arrays.
 *
 * @author <a href="mailto:boisvert@intalio.com">Alex Boisvert</a>
 * @version $Id: ByteArrayComparator.java,v 1.4 2002/05/31 06:33:20 boisvert Exp $
 */
public final class ByteArrayComparator
    implements Comparator, Serializable
{

    /**
     * Version id for serialization.
     */
    final static long serialVersionUID = 1L;


    /**
     * Compare two objects.
     *
     * @param obj1 First object
     * @param obj2 Second object
     * @return a positive integer if obj1 > obj2, 0 if obj1 == obj2,
     *         and a negative integer if obj1 < obj2
     */
     public int compare( Object obj1, Object obj2 )
     {
        if ( obj1 == null ) {
            throw new IllegalArgumentException( I18n.err( I18n.ERR_525 ) );
        }

        if ( obj2 == null ) {
            throw new IllegalArgumentException( I18n.err( I18n.ERR_526 ) );
        }

        return compareByteArray( (byte[]) obj1, (byte[]) obj2 );
     }


    /**
     * Compare two byte arrays.
     */
    public static int compareByteArray( byte[] thisKey, byte[] otherKey )
    {
        int len = Math.min( thisKey.length, otherKey.length );

        // compare the byte arrays
        for ( int i=0; i<len; i++ ) {
            if ( thisKey[i] >= 0 ) {
                if ( otherKey[i] >= 0 ) {
                    // both positive
                    if ( thisKey[i] < otherKey[i] ) {
                        return -1;
                    } else if ( thisKey[i] > otherKey[i] ) {
                        return 1;
                    }
                } else {
                    // otherKey is negative => greater (because MSB is 1)
                    return -1;
                }
            } else {
                if ( otherKey[i] >= 0 ) {
                    // thisKey is negative => greater (because MSB is 1)
                    return 1;
                } else {
                    // both negative
                    if ( thisKey[i] < otherKey[i] ) {
                        return -1;
                    } else if ( thisKey[i] > otherKey[i] ) {
                        return 1;
                    }
                }
            }
        }
        if ( thisKey.length == otherKey.length) {
            return 0;
        }
        if ( thisKey.length < otherKey.length ) {
            return -1;
        }
        return 1;
    }

}
