/*
 * $Id: gl.c,v 0.69 1995/04/20 22:10:21 zhao Beta $
 *
 *  A sample implementation of OpenGL canvas class
 *
 *   See ../DEMOS/gl.c for an example use of glcanvas.  
 *   See ../DEMOS/glwin.c for an example use of fl_glwinopen
 *
 */

#if defined(F_ID) || defined(DEBUG)
char *fl_id_gl = "$Id$";
#endif

#include "forms.h"
#include <GL/glx.h>
#include <GL/gl.h>

/*
 * Default GL canvas configuration. User can modify this default
 * using fl_set_glcanvas_default(FL_OBJECT *glcanvas, int *config)
 *
 */

static int glconfig[34] =
{
    GLX_RGBA, GLX_DEPTH_SIZE, 16, GLX_DOUBLEBUFFER,
    None
};

/* Initialize the OpenGL stuff before window creation */
static int
glx_init(FL_OBJECT * ob)
{
    XVisualInfo *vi;
    GLXContext context;

    /* query for openGL capabilities */
    if (!glXQueryExtension(fl_display, 0, 0))
    {
	fprintf(stderr, "GLCanvas: OpenGL not supported\n");
	return -1;
    }

    if (!(vi = glXChooseVisual(fl_display, fl_screen, glconfig)))
    {
	/* if can't get a visual, quit */
	fprintf(stderr, "GLCanvas: Can't get visual\n");
	return -1;
    }

    /* change canvas defaults */
    fl_set_canvas_visual(ob, vi->visual);
    fl_set_canvas_depth(ob, vi->depth);
    fl_set_canvas_colormap(ob, fl_create_colormap(vi, 1));

    context = glXCreateContext(fl_display, vi, None, GL_TRUE);

    if (!context)
    {
	fprintf(stderr, "Can't create GLX context!\n");
	return -1;
    }

    ob->u_vdata = context;
    return 0;
}

/*  This routine is called after a valid window is created. It simply
 *  binds the OpenGL context to the canvas window
 */
static int
glx_activate(FL_OBJECT * ob)
{
    glXMakeCurrent(fl_display, FL_ObjWin(ob), ob->u_vdata);
    M_warn("GLCanvas", "Binding context to 0x%lx", FL_ObjWin(ob));
    return 0;
}

static int
glx_cleanup(FL_OBJECT * ob)
{
    glXDestroyContext(fl_display, ob->u_vdata);
    return 0;
}


/* Interface routines */
FL_OBJECT *
fl_create_glcanvas(int type, FL_Coord x, FL_Coord y, FL_Coord w, FL_Coord h,
		   const char *label)
{
    FL_OBJECT *ob = fl_create_canvas(type, x, y, w, h, label);
    fl_modify_canvas_prop(ob, glx_init, glx_activate, glx_cleanup);
    return ob;
}

FL_OBJECT *
fl_add_glcanvas(int type, FL_Coord x, FL_Coord y, FL_Coord w, FL_Coord h,
		const char *label)
{
    FL_OBJECT *ob = fl_create_glcanvas(type, x, y, w, h, label);
    fl_add_object(fl_current_form, ob);
    return ob;
}

/* modify the default configuration of an opengl canvas */
void
fl_set_glcanvas_default(FL_OBJECT * ob, register int *config)
{
    register int *target = glconfig;

    while (*config != None)
	*target++ = *config++;
}

GLXContext
fl_get_glcanvas_context(FL_OBJECT * ob)
{
    return ob->u_vdata;
}


/*************************************************************
 * Open a top-level OpenGL window
 ************************************************************/

Window
fl_glwinopen(int *config, GLXContext * context, int w, int h)
{
    XVisualInfo *xvi;
    XSetWindowAttributes xswa;
    unsigned mask;
    Window win;

    if (!glXQueryExtension(fl_display, 0, 0))
	return 0;

    if (!(xvi = glXChooseVisual(fl_display, fl_screen, config)))
	return 0;

    *context = glXCreateContext(fl_display, xvi, None, GL_TRUE);

    xswa.colormap = fl_create_colormap(xvi, 1);
    xswa.border_pixel = 0;
    mask = CWColormap | CWBorderPixel;

    xswa.event_mask = ExposureMask;
    mask |= CWEventMask;

    win = XCreateWindow(fl_display, RootWindow(fl_display, fl_screen),
			0, 0, w, h, 0,
			xvi->depth, InputOutput,
			xvi->visual, mask, &xswa);
    if (win)
	glXMakeCurrent(fl_display, win, *context);

    return win;
}
