/* DB: Record List View
 * Copyright (C) 1998  by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "enum.h"
#include "db.h"
#include "callback.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

static Word TopVisibleRecord, LeftmostField, numColumnsShown;

static void
ListDrawTable(VoidPtr table, Word row, Word col, RectanglePtr bounds)
{
    Word recordNum;
    VoidHand recordHand;
    DBRecordType record;

    CALLBACK_PROLOGUE;

    if (row == 0) {
	FontID oldFont = FntSetFont(boldFont);
	WinDrawChars(dbInfo.fields[LeftmostField+col].name,
		     StrLen(dbInfo.fields[LeftmostField+col].name),
		     bounds->topLeft.x, bounds->topLeft.y);
	FntSetFont(oldFont);
	CALLBACK_EPILOGUE;
	return;
    }

    recordNum = (Word) TblGetRowID(table, row);
    GetRecord(recordNum, &record, &recordHand);
    /* we only support string type right now. ... hehe */
    WinDrawChars(record.field[LeftmostField+col],
		 StrLen(record.field[LeftmostField+col]),
		 bounds->topLeft.x, bounds->topLeft.y);
    MemHandleUnlock(recordHand);
    CALLBACK_EPILOGUE;
}

static void
UpdateFieldButtons(void)
{
    FormPtr form;
    Word leftIndex, rightIndex;

    form = FrmGetActiveForm();
    leftIndex = FrmGetObjectIndex(form, ctlID_ListView_LeftButton);
    rightIndex = FrmGetObjectIndex(form, ctlID_ListView_RightButton);

    if (LeftmostField > 0)
	FrmShowObject(form, leftIndex);
    else
	FrmHideObject(form, leftIndex);

    if (LeftmostField + numColumnsShown - 1 < dbInfo.numFields - 1)
	FrmShowObject(form, rightIndex);
    else
	FrmHideObject(form, rightIndex);
}

static void
ListUpdateScrollButtons(void)
{
    Word upIndex, downIndex, recordNum;
    SWord row;
    FormPtr form;
    TablePtr table;

    Boolean scrollUp, scrollDown;

    /* We can scroll up if the top record in the list view is not the first
     * record in the current category.
     */
    recordNum = TopVisibleRecord;
    scrollUp = DmSeekRecordInCategory(CurrentDB, &recordNum, 1, dmSeekBackward,
				      dmAllCategories) == 0;

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_ListView_Table);
    row = TblGetLastUsableRow(table);
    if (row != -1)
	recordNum = TblGetRowID(table, row);

    scrollDown = DmSeekRecordInCategory(CurrentDB, &recordNum, 1,
					dmSeekForward, dmAllCategories) == 0;

    upIndex = FrmGetObjectIndex(form, ctlID_ListView_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_ListView_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);
}

static void
ListLoadTable(TablePtr table)
{
    Word numRows, row, recordNum, col;

    numRows = TblGetNumberOfRows(table);
    recordNum = TopVisibleRecord;

    for (row = 1; row < numRows; row++) {
	if (DmSeekRecordInCategory(CurrentDB, &recordNum, 0, dmSeekForward,
				   dmAllCategories) != 0)
	    break;

	TblSetRowUsable(table, row, true);
	TblMarkRowInvalid(table, row);
	TblSetRowID(table, row, (DWord) recordNum);

	for (col = 0; col < numColumnsShown; col++) {
	    TblSetItemStyle(table, row, col, customTableItem);
	}

	recordNum++;
    }

    /* Mark the remainder of the rows as unusable. */
    while (row < numRows) {
	TblSetRowUsable(table, row, false);
	row++;
    }

    /* Make the scroll buttons reflect the table. */
    ListUpdateScrollButtons();
}

static void
ListInitTable(TablePtr table)
{
    RectangleType r;
    Word x, colnum, width;
    TblGetBounds(table, &r);

    /* Initialize the table. */
    TblSetRowUsable(table, 0, true);
    TblMarkRowInvalid(table, 0);
    x = colnum = 0;
    while (x < r.extent.x) {
	/* Stop processing if we reached the last column or field. */
	if (LeftmostField+colnum >= dbInfo.numFields || colnum >= 16)
	    break;

	width = dbInfo.fields[LeftmostField+colnum].colwidth;
	if (width > r.extent.x - x)
	    width = r.extent.x - x;
	if (LeftmostField+colnum == dbInfo.numFields-1
	    && width < r.extent.x - x)
	    width = r.extent.x - x;

	TblSetItemStyle(table, 0, colnum, customTableItem);
	TblSetColumnUsable(table, colnum, true);
	TblSetColumnWidth(table, colnum, width);
	TblSetCustomDrawProcedure(table, colnum, ListDrawTable);

	x += width;
	colnum++;
    }
    numColumnsShown = colnum;

    /* Make sure that the remainder of the columns are not usable. */
    while (colnum < 16) {
	TblSetColumnUsable(table, colnum, false);
	colnum++;
    }

    ListLoadTable(table);
    UpdateFieldButtons();
}

static void
ListScroll(DirectionType direction)
{
    UInt newTopVisibleRecord;

    newTopVisibleRecord = TopVisibleRecord;

    if (direction == down) {
	if (DmSeekRecordInCategory(CurrentDB, &newTopVisibleRecord,
				   1, dmSeekForward, dmAllCategories) != 0) {
	    newTopVisibleRecord = dmMaxRecordIndex;
	    DmSeekRecordInCategory(CurrentDB, &newTopVisibleRecord,
				   1, dmSeekBackward, dmAllCategories);
	}
    } else {
	if (DmSeekRecordInCategory(CurrentDB, &newTopVisibleRecord,
				   1, dmSeekBackward, dmAllCategories) != 0) {
	    newTopVisibleRecord = 0;
	    DmSeekRecordInCategory(CurrentDB, &newTopVisibleRecord,
				   1, dmSeekForward, dmAllCategories);
	}
    }

    if (TopVisibleRecord != newTopVisibleRecord) {
	FormPtr form;
	TablePtr table;

	TopVisibleRecord = newTopVisibleRecord;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_ListView_Table);
	ListLoadTable(table);
	TblRedrawTable(table);
    }
}

static void
AddDummyRecord(void)
{
    VoidHand recordH;
    UInt recordNum;
    CharPtr rec;
    ULong offset = 0;
    ULong alloced;

    alloced = dbInfo.numFields * 2;
    recordH = DmNewHandle(CurrentDB, alloced);
    rec = MemHandleLock(recordH);
    ErrFatalDisplayIf(rec == NULL, "rec is NULL");
    DmSet(rec, 0, alloced, 0);
    MemPtrUnlock(rec);

    recordNum = DmNumRecords(CurrentDB) + 1;
    DmAttachRecord(CurrentDB, &recordNum, (Handle) recordH, NULL);
    CurrentRecord = recordNum;
}

static Boolean
DoCommand(Word itemID)
{
    switch (itemID) {
    case menuitemID_DatabaseDesign:
	DesignNewDB = false;
	FrmGotoForm(formID_DesignView);
	return true;
    case menuitemID_EditFieldWidths:
	FrmGotoForm(formID_ListPropView);
	return true;
    case menuitemID_About:
	FrmAlert(alertID_About);
	return true;
    }
    return false;
}

Boolean
ListViewHandleEvent(EventPtr event)
{
    FormPtr form;
    TablePtr table;
    CharPtr name;

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();
	TopVisibleRecord = 0;
	LeftmostField = 0;
	table = GetObjectPtr(form, ctlID_ListView_Table);

	ListInitTable(table);
	FrmDrawForm(form);
	return true;

    case tblSelectEvent:
	table = event->data.tblSelect.pTable;
	if (event->data.tblSelect.row > 0) {
	    CurrentRecord = TblGetRowID(table, event->data.tblSelect.row);
	    FrmGotoForm(formID_EditView);
	}
	return true;

    case menuEvent:
	return DoCommand(event->data.menu.itemID);

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_ListView_DoneButton:
	    CloseDatabase();
	    FrmGotoForm(formID_Chooser);
	    return true;

	case ctlID_ListView_NewButton:
	    AddDummyRecord();
	    FrmGotoForm(formID_EditView);
	    return true;
	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_ListView_UpButton:
	    ListScroll(up);
	    break;

	case ctlID_ListView_DownButton:
	    ListScroll(down);
	    break;

	case ctlID_ListView_LeftButton:
	    form = FrmGetActiveForm();
	    table = GetObjectPtr(form, ctlID_ListView_Table);
	    LeftmostField--;
	    ListInitTable(table);
	    TblMarkTableInvalid(table);
	    TblRedrawTable(table);
	    break;

	case ctlID_ListView_RightButton:
	    form = FrmGetActiveForm();
	    table = GetObjectPtr(form, ctlID_ListView_Table);
	    LeftmostField++;
	    ListInitTable(table);
	    TblMarkTableInvalid(table);
	    TblRedrawTable(table);
	    break;
	}
	break;
    }

    return false;
}
